open Pi
open Net
open Data
open Crypto
open Formats_ciphersuites
open Formats 
open Conversions_lowlevel
open Conversions

open Db
//open Cell


type CipherState = 
  | BlockCipherState of bytes * bytes    // (key,iv)
  | StreamCipherState of stream_cipher_state

type ConnectionState = {
  cipher_state: CipherState;
  mk: bytes;
  seq_num: int; (* uint64 actually *)
  sparams: SecurityParameters; 
}

type ConnectionId = bytes * ConnectionEnd

type Connection = {
  net_conn: Net.conn;
  entity: ConnectionEnd;
  sessid: SessionID;
  crt_version: ProtocolVersion;
  read: ConnectionState;
  write: ConnectionState;
#if concrete
  buffer: bytes;
#endif
}

(*--- connectionsBegin *)
let connectionsMatch conn conn' = 
  if conn.sessid = conn'.sessid then
    if conn.crt_version = conn'.crt_version then
      if conn.read = conn'.write then
        if conn.write = conn'.read then
          ()
        else failwith "connection states do not match"	
      else failwith "connection states do not match"	
    else failwith "versions do not match"	
  else failwith "session ids do not match"
(*--- connectionsEnd *)

type event = 
  | Send' of ConnectionId * ConnectionEnd * bytes
  | Recv' of ConnectionId * ConnectionEnd * bytes

  | LeakedPayload of ConnectionEnd * bytes * Connection
  | LeakedConnection of Connection

  | Send of ConnectionEnd * bytes * Connection
  | Recv of ConnectionEnd * bytes * Connection

  | KeyLeak of bytes 

(*
(*--- recEventsBegin *)
  Send(entity, payload, conn)
  Recv(entity, payload, conn)
(*--- recEventsEnd *)
*)

(*
(* recEventsBegin *)
    Send'(id,entity,payload) 
    Recv'(id,entity,payload) 
(* recEventsEnd *)
*)

let tr = Pi.trace()


#if cv
(* corresponds to the Connected module in our papers *)


type keyentry = Keys of bytes * bytes

let msDb = Db.newdb (rand_guid()) 

let key_gen () = 
  let KeyExpansion = utf8 "key expansion" in
  let cr = noncetobytes (rand_nonce()) in
  let sr = noncetobytes (rand_nonce()) in
  let ms = noncetobytes (rand_nonce()) in
  let mk = noncetobytes (prf ms KeyExpansion (concat sr cr) 0) in
  let ek = noncetobytes (prf ms KeyExpansion (concat sr cr) 1) in
  mk, ek

let msgen ab = 
  let mk, ek = key_gen () in
  insertdb msDb ab (Keys (mk,ek))

let leak_msgen ab = 
  let mk, ek = key_gen () in
  Pi.log tr (KeyLeak(ab));      
  let ii = insertdb msDb ab (Keys (mk,ek)) in
  (mk,ek,ii)
    
let getKeys ab = 
  match selectdb msDb ab with
  | Some(Keys(mk,ek)) -> (mk,ek)
  | _ -> failwith "the key is compromised or is for the other role"


let select (a:bytes) (connid:ConnectionId) = 
  let mk,ek = getKeys connid in
  let id,entity = connid in
  let connstate = 
    { cipher_state = BlockCipherState(ek,empty_bstr); 
      mk = mk;
      seq_num = 0;
      sparams = {cipher_type = CT_block; bulk_cipher_algorithm = BCA_aes; mac_algorithm = MA_sha1; }
    } 
  in
  { net_conn = chan "fake"; 
    entity = entity; 
    sessid = empty_bstr; 
    crt_version = TLS1p0; 
    read = connstate; 
    write = connstate 
  }



#endif


#if fs
let debug = ref true
let set_debug d = debug := d

let sout = stdout
let rout = stdout
#endif






let chksecrecyChan = Pi.chan "chksecrecyChan"

#if cv
#else
let checkSecrecy conn payload = 
  let guess = Pi.recv chksecrecyChan in
  if guess = payload then 
    Pi.log tr (LeakedPayload(conn.entity,payload,conn))
  else 
    failwith "checksecrecy error"
#endif


let seqnumChan : bytes Pi.chan = Pi.chan "seqnumChan"






(*** initializing and maintaining connections ***)

let connDb = create()

let getConnection (connid: ConnectionId) = 
  select connDb connid

let storeConnection (connid: ConnectionId) conn = 
  insert connDb connid conn




#if concrete
let printCipherState cipher_state = 
  if !debug then
    match cipher_state with
    | BlockCipherState (key,iv) -> 
	begin
	  print_bytes "key" key;
	  print_bytes "iv" iv
	end
    | _ -> ()//failwith "not yet"

let printConnectionState str (conn_st:ConnectionState) = 
  if !debug then
    begin
      printf "%s\n" str;
      printCipherState conn_st.cipher_state;
      print_bytes "mk" conn_st.mk;
      printf "seq = %d\n\n" conn_st.seq_num
    end
#endif



let init_cipher_state cipher_type bulk_cipher_algorithm = 
  match cipher_type with
    | CT_block -> BlockCipherState (empty_bstr,empty_bstr)
    | CT_stream -> 
	begin
	  match bulk_cipher_algorithm with
	    | BCA_null -> StreamCipherState (stream_cipher_init (zeros 32))
	    | BCA_rc4 -> StreamCipherState (stream_cipher_init (zeros 32))
	    | _ -> failwith "[init_cipher_state] other stream ciphers not supported"
	end




let init_sp_null () = 
  { 
    bulk_cipher_algorithm = BCA_null;
    cipher_type = CT_stream;
    mac_algorithm = MA_null;
  }

let init_sp cs = 
  let (bca,ct,ma) = 
    match cs with 
    | TLS_NULL_WITH_NULL_NULL -> (BCA_null,CT_stream,MA_null)
    | TLS_RSA_WITH_AES_128_CBC_SHA -> (BCA_aes,CT_block,MA_sha1)
#if concrete
    | TLS_RSA_WITH_DES_CBC_SHA -> (BCA_des,CT_block,MA_sha1)
    | TLS_RSA_WITH_RC4_128_SHA -> (BCA_rc4,CT_stream,MA_sha1)
    | TLS_RSA_WITH_NULL_MD5 -> (BCA_null,CT_stream,MA_md5)
    | TLS_RSA_WITH_NULL_SHA -> (BCA_null,CT_stream,MA_sha1)
    | TLS_RSA_WITH_RC4_128_MD5 -> (BCA_rc4,CT_stream,MA_md5)
    | _ -> failwith "internal error"
#endif
  in
  { 
    bulk_cipher_algorithm = bca;
    cipher_type = ct;
    mac_algorithm = ma;
  }


let initConnState () = 
  { cipher_state = init_cipher_state CT_stream BCA_null; 
    mk = empty_bstr; 
    seq_num = 0;
    sparams = init_sp_null ();
  }

let updateConnState_cipher_state conn_st new_cipher_state =
  { cipher_state = new_cipher_state; 
    mk = conn_st.mk; 
    seq_num = conn_st.seq_num;
    sparams = conn_st.sparams;
  }

let updateConnState_seq_num conn_st new_seq = 
  { cipher_state = conn_st.cipher_state; 
    mk = conn_st.mk; 
    seq_num = new_seq;
    sparams = conn_st.sparams;
 }

let updateConnState_keys conn_st new_cipher_state new_mk = 
  { cipher_state = new_cipher_state; 
    mk = new_mk; 
    seq_num = conn_st.seq_num;
    sparams = conn_st.sparams;
  }

let initConn net_conn entity version = 
  let conn_st = initConnState () in
  { net_conn = net_conn;
    entity = entity;
    sessid = empty_bstr;
    crt_version = version;
    read = conn_st;
    write = conn_st;
#if concrete
    buffer = empty_bstr;
#endif
  }


let updateConn_version conn sid version = 
  { net_conn = conn.net_conn;
    entity = conn.entity;
    sessid = sid;
    crt_version = version;
    read = conn.read;
    write = conn.write; 
#if concrete   
    buffer = conn.buffer;
#endif
  }

let updateConn_buffer conn new_buffer = 
  { net_conn = conn.net_conn;
    entity = conn.entity;
    sessid = conn.sessid;
    crt_version = conn.crt_version;
    read = conn.read;
    write = conn.write; 
#if concrete   
    buffer = new_buffer;
#endif
  }

let updateReadConnectionState conn new_conn_state = 
  { net_conn = conn.net_conn;
    entity = conn.entity;
    sessid = conn.sessid;
    crt_version = conn.crt_version;
    read = new_conn_state;
    write = conn.write; 
#if concrete   
    buffer = conn.buffer;
#endif
  }

let updateWriteConnectionState conn new_conn_state = 
  { net_conn = conn.net_conn;
    entity = conn.entity;
    sessid = conn.sessid;
    crt_version = conn.crt_version;
    read = conn.read;
    write = new_conn_state;
#if concrete   
    buffer = conn.buffer;
#endif
  }

let add1 x = 
#if fs
  x + 1
#else
  x
#endif

let updateReadSeqNum conn =
  let conn_state = conn.read in
  let conn_state = updateConnState_seq_num conn_state (add1 conn_state.seq_num) in
  updateReadConnectionState conn conn_state

let updateWriteSeqNum conn =
  let conn_state = conn.write in
  let conn_state = updateConnState_seq_num conn_state (add1 conn_state.seq_num) in
  updateWriteConnectionState conn conn_state


let get_read_seq_num conn =
  let rconnst = conn.read in
  bytes_of_seq rconnst.seq_num

let get_write_seq_num conn =
  let wconnst = conn.write in
  bytes_of_seq wconnst.seq_num


(*** sending ***)

let get_block_size bulk_cipher_algorithm = 
  match bulk_cipher_algorithm with
#if concrete
    | BCA_des -> 8
#endif
    | BCA_aes -> 16
    | _ -> failwith "[get_block_size] N/A or not yet supported"

(* do you have a simpler formula? *)
(* the ouput should be 0 -> 7, 1 -> 6, 2 -> 5, ..., 6 -> 1, 7 -> 8 when  bs=8 *)
let get_pad_len bs msg_len = 
//  (2*bs - 2 - (msg_len % bs)) % bs + 1
#if fs
  let r = msg_len % bs in
  if r <> (bs-1) then (bs-1) - r else bs
#else
  0 
#endif
      
let add x y = 
#if fs
  x + y
#else
  0
#endif




(* the actual padding is added by the encryption algorithm; here we just add the padding length byte *)
(* there is a trick here, we'll explain it sometime *)
(* we do not have arbitray length padding yet *)
let compute_blockcipherinput ver bca msg msg_len = 
  let bs = get_block_size bca in
  let prefix_len, prefix = 
#if concrete
    if PV2ver ver = TLS1p1 then
      bs, mkRandom bs
    else
      0, empty_bstr
#else
      0, empty_bstr
#endif
  in
  let pad_len = get_pad_len bs (add prefix_len msg_len) in
  let pad_len_byte = bytes_of_int 1 pad_len in
  let output_len = add (add (add prefix_len msg_len) 1) pad_len in 
  let plaintext = concat3 prefix msg pad_len_byte in
  (output_len,plaintext)    


let compute_mac_ssl3_md5 mac_secret bseq bct len payload = 
  let mmsg = concat6 mac_secret ssl_pad1_md5 bseq bct (bytes_of_int 2 len) payload in
  md5 (concat mac_secret (concat ssl_pad2_md5 (md5 mmsg)))

let compute_mac_ssl3_sha1 mac_secret bseq bct len payload = 
  let mmsg = concat6 mac_secret ssl_pad1_sha1 bseq bct (bytes_of_int 2 len) payload in
  sha1 (concat mac_secret (concat ssl_pad2_sha1 (sha1 mmsg)))

let compute_mac_md5 mac_secret bseq bct bver len payload = 
  let mmsg = concat5 bseq bct bver (bytes_of_int 2 len) payload in
  hmacmd5 (msymkey mac_secret) mmsg

let compute_mac_sha1 mac_secret bseq bct bver len payload = 
  let mmsg = concat5 bseq bct bver (bytes_of_int 2 len) payload in
  mactobytes (hmacsha1 (msymkey mac_secret) mmsg)


let compute_mac version mac_algorithm mac_secret bseq bct bver len payload =
#if concrete
  match mac_algorithm, PV2pmver version with 
  | MA_sha1, PM_SSLv3 -> compute_mac_ssl3_sha1 mac_secret bseq bct len payload
  | MA_md5, PM_SSLv3 -> compute_mac_ssl3_md5  mac_secret bseq bct len payload
  | MA_md5, _ -> compute_mac_md5 (bytestononce mac_secret) bseq bct bver len payload
  | MA_sha1, _ -> compute_mac_sha1 (bytestononce mac_secret) bseq bct bver len payload
  | MA_null,_ -> empty_bstr
#else
  match mac_algorithm with
  | MA_sha1 -> compute_mac_sha1 (bytestononce mac_secret) bseq bct bver len payload
  | MA_null -> empty_bstr
  | _ -> failwith "internal error"
#endif






#if concrete
let compute_new_iv version bulk_cipher_algorithm ciphertext = 
  let bs = get_block_size bulk_cipher_algorithm in
  if (PV2ver version) = TLS1p1 then
    mkRandom bs
  else
    get_last_block bs ciphertext
#endif

let encdec_block version bulk_cipher_algorithm key iv ciphertext = 
  let new_iv = 
#if concrete
    compute_new_iv version bulk_cipher_algorithm ciphertext 
#else
    iv
#endif
  in 
  (BlockCipherState (key,new_iv), ciphertext)

let encrypt version cipher_type bulk_cipher_algorithm cipher_state input = 
  match cipher_type, bulk_cipher_algorithm, cipher_state with
   | CT_stream, BCA_null, _ -> (cipher_state,input)
#if concrete
   | CT_stream, BCA_rc4, StreamCipherState st -> 
			let output = stream_cipher_encrypt input st in
		        (StreamCipherState st, output)
#endif
   | CT_stream, _, _ -> failwith "internal error"
#if concrete
   | CT_block, BCA_des, BlockCipherState (key,iv) -> 
	   encdec_block version bulk_cipher_algorithm key iv (des_encrypt (symkey (bytestononce key)) iv input)
#endif
   | CT_block, BCA_aes, BlockCipherState (key,iv) -> 
       let output = blockstobytes(aes_encrypt_wiv (symkey (bytestononce key)) iv (bytestoblocks input)) in
       encdec_block version bulk_cipher_algorithm key iv output
   | _,_,_ -> failwith "internal error"

let decrypt_ext version cipher_type bulk_cipher_algorithm cipher_state input = 
  match cipher_type, bulk_cipher_algorithm, cipher_state with
   | CT_stream, BCA_null, _ -> (cipher_state,input)
#if concrete
   | CT_stream, BCA_rc4, StreamCipherState st -> 
			let output = stream_cipher_encrypt input st in
		        (StreamCipherState st, output)
#endif
   | CT_stream, _, _ -> failwith "internal error"
#if concrete
   | CT_block, BCA_des, BlockCipherState (key,iv) -> 
       let new_iv = compute_new_iv version bulk_cipher_algorithm input in 
       (BlockCipherState (key,new_iv), des_decrypt (symkey (bytestononce key)) iv input)
#endif
   | CT_block, BCA_aes, BlockCipherState (key,iv) -> 
       let new_iv = 
#if concrete
         compute_new_iv version bulk_cipher_algorithm input
#else
         iv
#endif
       in 
       (BlockCipherState (key,new_iv), (blockstobytes (aes_decrypt_wiv (symkey (bytestononce key)) iv (bytestoblocks input))))
   | _,_,_ -> failwith "internal error"


#if concrete
let max_TLSPaintext_fragment_length = 1<<<14
#else
let max_TLSPaintext_fragment_length = 1
#endif

let compute_cipherinput version cipher_type bca p_and_m p_and_m_len = 
  match cipher_type with
  | CT_stream -> p_and_m_len, p_and_m
  | CT_block -> compute_blockcipherinput version bca p_and_m p_and_m_len  


(* payload --fragmentation--> plaintext -> compressed -> ciphertext *)
let send_record (tlsconn:Connection) (ct:ContentType) (payload:bytes) bseq = 
  (* fragmentation *)
  let len = length payload in
  (* "The length of a plaintext should not exceed 2^14." *)
#if fs
  if len > max_TLSPaintext_fragment_length then 
    failwith "internal error";
#endif
  (* compression *)
  (* mac&encrypt *)  
  let conn_state = tlsconn.write in
  let sp = conn_state.sparams in
  let ver = tlsconn.crt_version in
  let bca = sp.bulk_cipher_algorithm in
  let bct = bytes_of_ContentType ct in
  let bver = (bytes_of_ProtocolVersion ver) in
  let mac = compute_mac ver sp.mac_algorithm conn_state.mk bseq bct bver len payload in
  let p_and_m = concat payload mac in
  let p_and_m_len = length p_and_m in
  let record_len, cipherinput = compute_cipherinput ver sp.cipher_type bca p_and_m p_and_m_len in
  let new_cipher_state,ciphertext = encrypt ver sp.cipher_type bca conn_state.cipher_state cipherinput in
#if concrete
  if (length ciphertext) <> record_len then
    failwith (Printf.sprintf "something's wrong with the ciphertext length (it is %d, should be %d)" (length ciphertext) record_len);
#endif
  let header = concat3 bct bver (bytes_of_int 2 record_len) in
  let record = concat header ciphertext in

  (* update conn_state *)
  let conn_state = updateConnState_cipher_state conn_state new_cipher_state in
  let tlsconn = updateWriteConnectionState tlsconn conn_state in
  let net_conn = tlsconn.net_conn in
#if concrete
  if !debug then
    begin fprint_bytes sout ">>>Sent" record;fprintf sout "\n" end;
#endif
  tlsconn,record



(*** receiving ***)

(* returns 1 + the padding length (i.e. including the padding length byte) *)
let get_total_padding_length (ver:version) (msg:bytes) = 
#if concrete
  let lb = Bytearray.get msg (length msg - 1) in
  let l = Byte.to_int lb in
  (* in SSL3 the padding is random *)
  let wrong_padding = ref false in  
  if ver <> SSLv3 then
    for i = 1 to l do
      if (Bytearray.get msg (length msg - 1 - i)) <> lb then
	(* in TLS1.1 we don't fail now, we fail later, see sec.6.2. Implementation Note *)
	if  ver <> TLS1p1 then
	  failwith "decryption failed"
	else
	  wrong_padding := true
    done;
  if !wrong_padding then
    0
  else
    l + 1
#else
  0
#endif

#if concrete
let get_hash_size ma = 
  match ma with
  | MA_null -> 0
  | MA_sha1 -> 20
  | MA_md5 -> 16
#endif

let compute_payload version cipher_type bca ma (plaintext:bytes) = 
#if concrete
  let hash_size = get_hash_size ma in
  let length = length plaintext in
  let ver = PV2ver version in
  let i,payload_len = 
    match cipher_type with 
      | CT_stream -> (0, length - hash_size)
      | CT_block -> 
	  let plen = length - (hash_size + get_total_padding_length ver plaintext) in   
	  let bs = get_block_size bca in
	  if ver = TLS1p1 then
	    (bs, plen-bs)
	  else
	    (0, plen)
  in
  let payload = Bytearray.sub plaintext i payload_len in
  let mac = Bytearray.sub plaintext (i+payload_len) hash_size in
  (payload, mac)
#else
  match cipher_type with 
  | CT_stream -> iconcat plaintext
  | CT_block -> 
      let prefix, content, padding = iconcat3 plaintext in
      iconcat content
#endif



(* we return the records in sequence, even when not encrypted *)
#if concrete

let print_recv buf =
  if !debug then
    begin
      if length buf < 518 then
	print_bytes "<<<Received" buf
      else
	Printf.printf "<<<Received (%d)" (length buf);
      print_newline()
    end
  else
    ()
      
let recvRecord conn = 
  let buf = conn.buffer in
  let buf = 
    if length buf = 0 then 
      begin
	let buf = Net.recvl conn.net_conn in
	if buf.Length = 0 then
	  failwith "empty message";
	print_recv buf;
	buf
      end
    else
      buf
  in
  if length buf < 5 then 
    failwith "decode error";
  let rec_len = 5 + int_of_bytes 2 (Bytearray.sub buf 3 2) in
  let rec rec_recv conn (buf:bytes) = 
    if length buf < rec_len then 
      begin
	let msg = Net.recvl conn.net_conn in
	if msg.Length = 0 then
	  failwith "empty message";
	print_recv buf;
	rec_recv conn (concat buf msg)
      end
    else
      buf
  in
  let buf = rec_recv conn buf in
  let record = Bytearray.sub buf 0 rec_len in
  let buf = Bytearray.sub buf rec_len (length buf - rec_len) in
  let conn = updateConn_buffer conn buf in
    conn, record
#else
let recvRecord conn = 
  let msg =  Net.recvl conn.net_conn in
#if fs
  fprint_bytes rout "<<<Received" msg;
#endif
  conn,msg
#endif

let hmacVerify connst maced recvmac = 
  let sp = connst.sparams in
  let mac_alg = sp.mac_algorithm in
  let mk = connst.mk in
  match mac_alg with
  | MA_null -> true
  | MA_sha1 -> hmacsha1_verify (msymkey (bytestononce mk)) maced (bytestomac recvmac) 
#if concrete
  | MA_md5 ->  hmacmd5_verify (msymkey (bytestononce mk)) maced (bytestomac recvmac) 
#else
  | _ -> failwith "not available otherwise"
#endif





let recv_partial (tlsconn: Connection) (input: bytes) bseq = 
  let (bct, bver, blen, msg) = parseRecord input in
  let rct,rver,len,msg = 
      (ContentType_of_bytes bct, ProtocolVersion_of_bytes bver, int_of_bytes 2 blen, msg)
  in
  print_tls rct rver len;

  (* decrypting *)
  let conn_state = tlsconn.read in
  let sp = conn_state.sparams in
  let ver = tlsconn.crt_version in
  let bca = sp.bulk_cipher_algorithm in
  let new_cipher_state, plaintext = decrypt_ext ver sp.cipher_type bca conn_state.cipher_state msg in
  let payload, recvmac = compute_payload ver sp.cipher_type bca sp.mac_algorithm plaintext in

  (* checking the mac *)
  let ourmac = compute_mac ver sp.mac_algorithm conn_state.mk bseq bct bver (length payload) payload in
  
  if recvmac = ourmac then
    ()
  else 
    failwith "bad record mac";

  (* decompression *)
  (* defragmentation *)

  (* update conn_state *)
  let conn_state = updateConnState_cipher_state conn_state new_cipher_state in
  let tlsconn = updateReadConnectionState tlsconn conn_state in
  (tlsconn,rct,rver,payload)






(* code for condensed verifyPayload *)

let decrypt ver connst ciphertext = 
  let sp = connst.sparams in
  let new_cipher_state, plaintext = decrypt_ext ver sp.cipher_type sp.bulk_cipher_algorithm connst.cipher_state ciphertext in
  let connst = updateConnState_cipher_state connst new_cipher_state in
  connst, plaintext
    

let parsePlaintext ver connst plaintext = 
  let sp = connst.sparams in
  compute_payload ver sp.cipher_type sp.bulk_cipher_algorithm sp.mac_algorithm plaintext




let getAbstractValues bct brver blen = 
  (ContentType_of_bytes bct, ProtocolVersion_of_bytes brver, int_of_bytes 2 blen)



let checkContentType (ct:ContentType) (rct:ContentType) (payload:bytes) = 
  if rct = ct then
    ()
#if concrete
  else
    begin
      if rct = CT_alert then
	begin
	  if !debug then
	    printf "\nThe following alert has been received: %s\n" (string_of_Alert (Alert_of_bytes payload));
          failwith "[verify_payload] we stop"
	end
      else if rct = CT_application_data then
	if !debug then
	  begin
	    printf "---\n%s\n---\n\n" (istr (iutf8 payload));
	    failwith "wrong content-type"
	  end
    end
#else
  else
    failwith "wrong content-type"
#endif


(* ugly, but needed: for symbolic verification the sequence number is
   received from the adversary *) 
let bytes_of_seq seq_num = 
#if concrete
  bytes_of_seq seq_num
#else
  Pi.recv seqnumChan
#endif



(* only working for TLS1p0, not for SSLv3 because of the non standard mac *)
(* also, we cannot use it for the handshake because of the version test! *)
(*--- getPayloadBegin *)
let verifyPayload (conn:Connection) (ct:ContentType) (input:bytes)= 
  let (bct, bver, blen, ciphertext) = parseRecord input in
  let rct, rver, rlen = getAbstractValues bct bver blen in
  let ver = conn.crt_version in
  if rver = ver then
    let connst = conn.read in
    let connst, plaintext = decrypt ver connst ciphertext in
    let payload, recvmac = parsePlaintext ver connst plaintext in
    let len = bytes_of_int 2 (length payload) in
    let bseq = bytes_of_seq connst.seq_num in
    let maced = concat5 bseq bct bver len payload in
    if hmacVerify connst maced recvmac = true then
      begin
        checkContentType ct rct payload;
        let conn = updateReadConnectionState conn connst in
        (conn, payload)
      end
    else failwith "bad record mac"
  else failwith "bad version"
(*--- getPayloadEnd *)


(* working for SSLv3 too *)
let verify_payload (tlsconn: Connection) (ct: ContentType) input seq_num (checkver: bool) = 
  let tlsconn,rct,rver,payload = recv_partial tlsconn input seq_num in
  checkContentType ct rct payload;
(* we should perform this test only after (strictly) receiving the ServerHello *)
#if concrete
  if checkver = true then
#endif
    if rver = tlsconn.crt_version then
      ()
    else
      failwith "wrong version number";
  tlsconn,payload






(* send/recv unprotected payloads: when we know a priori that the ciphersuite is TLS_NULL_WITH_NULL_NULL *)
(* we'll only use them symbolically, and hence only for the handshake *)

let bldRecord ver ct payload =
  let len = length payload in
  let header = concat3 
      (bytes_of_ContentType ct) 
      (bytes_of_ProtocolVersion ver) 
      (bytes_of_int 2 len) in
  concat header payload

let send_null netconn ver ct payload = 
  let record = bldRecord ver ct payload in
#if concrete
  if !debug then
    begin fprint_bytes sout ">>>Sent" record;fprintf sout "\n" end;
#endif
  Net.sendl netconn record


let ibldRecord record =
  let (bct, bver, blen, payload) = parseRecord record in
  let rct,rver,len = (ContentType_of_bytes bct, ProtocolVersion_of_bytes bver, int_of_bytes 2 blen) in
  print_tls rct rver len;
  rct, payload
    
(* we should also check the version *)
let recv_null netconn ct = 
  let record = Net.recvl netconn in
  let rct,payload = ibldRecord record in
  if rct = ct then
    ()
  else
    failwith "wrong content-type";
  payload

let recv_ct conn ct checkver =
  let conn, input = recvRecord conn in
  let seq_num = get_read_seq_num conn in
  let conn, msg = verify_payload conn ct input seq_num checkver in
  let conn = updateReadSeqNum conn in
  conn, msg

let recv_hs conn checkver = recv_ct conn CT_handshake checkver
let recv_alert conn = recv_ct conn CT_alert false
let recv_ccs conn = recv_ct conn CT_change_cipher_spec true


let send_ct conn ct msg =
  let seq_num = get_write_seq_num conn in
  let conn, record = send_record conn ct msg seq_num in
  Net.sendl conn.net_conn record;
  let conn = updateWriteSeqNum conn in
  conn

let send_hs conn msg = send_ct conn CT_handshake msg
let send_alert conn msg = send_ct conn CT_alert msg
let send_ccs conn msg = send_ct conn CT_change_cipher_spec msg


let send connid msg = 
  let conn = getConnection connid in
  #if concrete
    let seq_num = get_write_seq_num conn in
  #else
    let seq_num = Pi.recv seqnumChan in
  #endif
  #if cv 
    #if thm_record_auth_comp
      Pi.log tr (Send'(connid,conn.entity,msg));
    #endif
  #else
    Pi.log tr (Send(conn.entity,msg,conn));
  #endif
  let conn,record = send_record conn CT_application_data msg seq_num in
  #if concrete
    storeConnection connid conn;
  #endif
  Net.sendl conn.net_conn record



#if tlsonly

(* condensed recv *)
(*--- recvBegin *)
let recv (connid:ConnectionId) = 
  let conn = getConnection connid in
  let conn, input = recvRecord conn in 
  let conn, msg = verifyPayload conn CT_application_data input in
  log tr (Recv(conn.entity, msg, conn));
  let conn = updateReadSeqNum conn in
  storeConnection connid conn;
  msg
(*--- recvEnd *)

#else

(* the more general recv *)
let recv connid  =
  let conn = getConnection connid in
  let conn, input = recvRecord conn in
  #if concrete
    let seq_num = get_read_seq_num conn in
  #else
    let seq_num = Pi.recv seqnumChan in
  #endif      

  #if cv (* we could also use this version concretely and symbolically *)
    let conn, msg = verifyPayload conn CT_application_data input in
  #else (* the more general version *)
    let conn, msg = verify_payload conn CT_application_data input seq_num true in
  #endif

  let conn = updateReadSeqNum conn in

  let entity = conn.entity in
  #if cv
    #if thm_record_auth_comp
      Pi.log tr (Recv'(connid,entity,msg));
    #endif
  #else
    Pi.log tr (Recv(entity,msg,conn));
  #endif

  #if concrete
    storeConnection connid conn;
  #endif

  msg

#endif





let send_fresh_payload connid = 
  let secr = rand_nonce () in 
  let payload = noncetobytes secr in
  send connid payload
  #if cv
    #if thm_record_secr_comp
      ;secret_nonce secr 
    #endif
  #else
    ;let conn = getConnection connid in
    checkSecrecy conn payload 
  #endif


#if cv
#else
(*--- leakConnBegin *)
let leakConnection connid = 
  let conn = getConnection connid in
  log tr (LeakedConnection(conn));
  conn
(*--- leakConnEnd *)
#endif

