open Data
open Formats_ciphersuites 
open Formats
open Conversions_lowlevel
open Conversions
open Handshake
open Record
open App_passwd
#if concrete
open App_http
#endif




let defaultUri = "https://127.0.0.1:8081"


(************************************************)
(*** arguments ***)

let bad_arg str = 
  sprintf "unrecongnized argument: %s\ntry: [client -help | server -help] for the list of arguments\n" str

let reconnect = ref false
let reconnect_arg = ("-reconnect", Arg.Set reconnect, 
		   "resume the connection")

let rehandshake_par = ref false
let rehandshake_arg = ("-rehandshake", Arg.Set rehandshake_par, 
		   "drop and then resume the connection with the same session id")

let passwd = ref false
let passwd_arg = ("-passwd", Arg.Set passwd, 
		   "perform a simple password authentication test")


#if concrete

let uri = ref defaultUri
let f_uri str = uri := str
let connect_arg = ("-connect", Arg.String f_uri, 
		   "who to connect to (default is https://localhost:8081)")


let ver_min = ref (ver2PV SSLv3)
let parse_ver_min str = 
  ver_min := ver2PV (
    match str with
    | "ssl2" -> SSLv2
    | "ssl3" -> SSLv3
    | "tls1p0" -> TLS1p0
    | "tls1p1" -> TLS1p1
    | _ -> raise (Arg.Bad str))

let ver_max = ref (ver2PV TLS1p1)
let parse_ver_max str = 
  ver_max := ver2PV (
    match str with
    | "ssl2" -> SSLv2
    | "ssl3" -> SSLv3
    | "tls1p0" -> TLS1p0
    | "tls1p1" -> TLS1p1
    | _ -> raise (Arg.Bad str))


let ver_min_arg = ("-ver_min", Arg.String parse_ver_min, 
		   "minimal supported version (i.e. string::=ssl3,tls1p0,tls1p1; default is ssl3)")

let ver_max_arg = ("-ver_max", Arg.String parse_ver_max, 
		   "maximal supported version (i.e. string::=ssl3,tls1p0,tls1p1; default is tls1p1)")



let ccs_list=
    [ TLS_RSA_WITH_AES_128_CBC_SHA;
      TLS_RSA_WITH_RC4_128_MD5;
      TLS_RSA_WITH_RC4_128_SHA;       
      TLS_RSA_WITH_DES_CBC_SHA;
      TLS_RSA_WITH_NULL_MD5;               
      TLS_RSA_WITH_NULL_SHA;
      TLS_NULL_WITH_NULL_NULL;
    ]

let csl = ref ccs_list
let parse_csl str = 
  csl := let cs = 
    match str with     
    | "TLS_RSA_WITH_AES_128_CBC_SHA" -> TLS_RSA_WITH_AES_128_CBC_SHA
    | "TLS_RSA_WITH_RC4_128_MD5" -> TLS_RSA_WITH_RC4_128_MD5
    | "TLS_RSA_WITH_RC4_128_SHA" -> TLS_RSA_WITH_RC4_128_SHA
    | "TLS_RSA_WITH_DES_CBC_SHA" -> TLS_RSA_WITH_DES_CBC_SHA
    | "TLS_NULL_WITH_NULL_NULL" -> TLS_NULL_WITH_NULL_NULL
    | "TLS_RSA_WITH_NULL_MD5" -> TLS_RSA_WITH_NULL_MD5
    | "TLS_RSA_WITH_NULL_SHA" -> TLS_RSA_WITH_NULL_SHA
    | _ -> raise (Arg.Bad str)
  in [cs] (* ^ !ccs_list *)



let csl_arg = ("-cs", Arg.String parse_csl, 
		   "the ciphersuite (i.e. string::=TLS_RSA_WITH_AES_128_CBC_SHA, etc) (see readme.txt for the whole list)")


let http = ref false
let http_arg = ("-http", Arg.Set http, 
		   "make an HTTP request and print the response")


let close_notify = ref false
let close_arg = ("-close", Arg.Set close_notify, 
		   "if used, the connection is closed by sending a close_notify alert")

let wait_close = ref false
let wait_close_arg = ("-wait", Arg.Set wait_close, 
		   "if used, after sending a close_notify alert we wait for a response")

let certname = ref "localhost"
let f_certname str = certname := str
let certname_arg = ("-certname", Arg.String f_certname, 
		   "certificate to use from the store (default is localhost)")

(*
let scenario = ref false
let scenario_arg = ("-scenario", Arg.Set scenario, 
		     "connect, various scenarios:\
		      1.reconnect and do a HTTP exchange (i.e.-reconnect and -http)\
                      2.\
                      3.")
*)

let ctestall = ref false
let ctestall_arg = ("-testall", Arg.Set ctestall, 
		   "test the client with all combinations of versions and ciphersuites against a list of servers")


let downgrade = ref false
let downgrade_arg = ("-downgrade", Arg.Set downgrade, 
		   "test version downgrading (use `ver_max' for setting the initial version, and `ver_min' for setting the version to downgrade to)")


let current = ref (-1)


let anon_fun str =
  raise (Arg.Bad (bad_arg str))



let client_speclist = [connect_arg; reconnect_arg; rehandshake_arg; http_arg; passwd_arg;
		       ver_min_arg; ver_max_arg; csl_arg; close_arg; wait_close_arg;
		       ctestall_arg; downgrade_arg]
let server_speclist = [http_arg; passwd_arg; ver_min_arg; ver_max_arg; csl_arg; certname_arg]


let usage_str = "usage: client args | server args"

let entity = ref Client
let select_entity str =
  let new_argv = Array.sub Sys.argv 2 ((Array.length Sys.argv) - 2) in
  try 
  entity := 
      match str with
      | "client" -> 
	  begin
	    Arg.parse_argv current new_argv client_speclist anon_fun "usage: client args";
	    Client
	  end
      | "server" -> 
	  begin
	    Arg.parse_argv current new_argv server_speclist anon_fun "usage: server args";
	    Server
	  end
      | _ -> raise (Arg.Bad (sprintf "unrecongnized argument: %s\n%s\n" str usage_str))
  with 
  | Arg.Bad str -> begin printf "%s" str; exit 0 end
  | Arg.Help str -> begin printf "%s" str; exit 0 end




let connect net_conn = 
  connect_ext net_conn !ver_min !ver_max !csl !uri




(* not used for the moment*)
let unlimApp connid firstrecv = 
  if firstrecv then
    let msg = recv connid in
    ();
  while true do 
    System.Threading.Thread.Sleep(1000);
    let n = Crypto.mkNonce256() in
    send connid n;
    let msg = recv connid in
    ()
  done
    
let resumeApp connid sessid () = 
   let connid,_ = rehandshake connid sessid in
   unlimApp connid false 
(****************)


(* not used for the moment *)
let unlimClientHttp connid () = 
  while true do 
    //System.Threading.Thread.Sleep(1000);
    clientHttpSendReq connid !uri;
    clientHttpRecvReq connid 
  done

let clientUnlimHttp () = 
  let conn = Net.connect !uri in 
  let connid,sessid = connect conn in
  Pi.fork (unlimClientHttp connid)
(****************)



(* trying rehandshake *)
(*
let client_rehandshake uri http = 
  let conn = Net.connect uri in 
  let connid,sessid = connect conn uri in
  if http then
    clientHttp connid uri;
  let connid,_ = rehandshake connid sessid in
  if http then
    clientHttp connid uri
  else
    ()
*)

(* trying resumption *)
(*
let client_resume uri http = 
  let net_conn = Net.connect uri in 
  let _ = Net.recvl net_conn in
  let connid,sessid = connect net_conn uri in
  if http then
    clientHttp connid uri;
  let _ = close connid false in
  let net_conn = Net.connect uri in 
  // new:
  let sess = getSession sessid in
  let conn = initConn net_conn sess.sh.server_version in
  let connid,_ = resume_ext conn sess ccs_list None in
//(Some (ver2PV SSLv3)) in
  //let connid,_ = resume conn sessid in
  if http then
    clientHttp connid uri
  else
    ()
*)

(* quite interestingly, it seems that
some servers accept resumption but not rehandshake (like www.loria.fr),
while others accept rehandshake but not resumption (like www.google.com) 
*)




  
let general_scenario ver_min ver_max ver_res ccs_list server_id http close_conn wait_close = 
    let net_conn = Net.connect server_id in
    let connid, sessid = connect_ext net_conn ver_min ver_max ccs_list server_id in
    (* do a http request/response? *)
    if http then
      httpGet connid server_id;
    (* close the connection? *)
    let net_conn, conn =
      if close_conn then
        begin 
	  close connid wait_close; 
	  Net.close net_conn;	  
          let net_conn = Net.connect server_id in
	  let sess = getcSession sessid in
	  (net_conn, initConn net_conn Client sess.sh.server_version)
	end
      else
	(net_conn, getConnection connid)
    in
    let sess = getcSession sessid in
    let connid, newsessid = resume_ext conn sess ccs_list ver_res in
    if http then
      httpGet connid server_id;
    connid, sessid, newsessid


let server_ver_downgrade ver_full ver_res ccs_list server_id http close_conn wait_close = 
  try
    let connid, sessid, newsessid = general_scenario ver_full ver_full ver_res ccs_list server_id http close_conn wait_close in
    if sessid <> newsessid then
      begin
	failwith "The server didn't accept to resume."
      end
    Printf.printf "\nVersion downgrade succeeded.\n"
  with e -> 
      begin  
	Printf.printf "\nVersion downgrade failed.\n";
	Printf.printf "%s\n" (e.ToString());
      end

let server_ver_downgrade2 () =
  try
    let connid, sessid, newsessid = general_scenario !ver_max !ver_max (Some !ver_min) !csl !uri !http !close_notify !wait_close in
    if sessid <> newsessid then
      failwith "The server didn't accept to resume.";
    Printf.printf "\nVersion downgrade succeeded.\n"
  with e -> 
      begin  
	Printf.printf "\nVersion downgrade failed.\n";
	Printf.printf "%s\n" (e.ToString());
      end

let client_resume2 () = 
  ignore (general_scenario !ver_min !ver_max None !csl !uri !http !close_notify !wait_close)




(* the only difference between version downgrade and resume function is: 
 - in resume, ver_min, ver_max are as such for the full handshake;
resumption is performed using the just established version 
 - in downgrade, ver_min, ver_max represent the version to be used in the
resumption protocol, and the full handshake respectively
*)

(* testing server version downgrading *)
(*
let server_ver_downgrade ver_full ver_res server_id http close_conn wait_close = 
  try
    let net_conn = Net.connect server_id in
    let connid, sessid = connect_ext net_conn ver_full ver_full ccs_list server_id in
    (* need to do a http request/response? *)
    if http then
      clientHttp connid server_id;
    (* need to close the connection? *)
    let net_conn, conn =
      if close_conn then
        begin 
	  close connid wait_close; 
	  Net.close net_conn;	  
          let net_conn = Net.connect server_id in
	  (net_conn, initConn net_conn Client ver_res)
	  //(net_conn, initConn net_conn sess.sh.server_version)
	end
      else
	(net_conn, getConnection connid)
    in
    let sess = getSession sessid in
    let connid, newsessid = resume_ext conn sess ccs_list (Some ver_res) in
    if sessid <> newsessid then
      begin
	failwith "The server didn't accept to resume."
      end
    clientHttp connid server_id;
    Printf.printf "\nVersion downgrade succeeded.\n"
  with e -> 
      begin  
	Printf.printf "\nVersion downgrade failed.\n";
	Printf.printf "%s\n" (e.ToString());
      end
*)


(*** code for test cases generation ***)

let ver_list = [SSLv3; TLS1p0; TLS1p1]

let ccslist=
    [ TLS_RSA_WITH_AES_128_CBC_SHA;
      TLS_RSA_WITH_RC4_128_MD5;
      TLS_RSA_WITH_RC4_128_SHA;       
      TLS_RSA_WITH_DES_CBC_SHA;
      (* no known server accepts these: *)
      //TLS_RSA_WITH_NULL_MD5;               
      //TLS_RSA_WITH_NULL_SHA;
      //TLS_NULL_WITH_NULL_NULL;
    ]


let server_list = [
   "www.microsoft.com";
   "www.google.com";
   "www.yahoo.com";
   "www.google.fr";   
   "www.loria.fr";
   "www.inria.fr";
   "www.msr-inria.inria.fr";
   "wiki.msr-inria.inria.fr";
   "interne.futurs.inria.fr";
   "particuliers.secure.lcl.fr";
   "www.facebook.com";
   //"test.gnutls.org:5556/"; (* it asks for client authentication; we don't support it *)
   ]

(* conclusions after a simple run (client, single full handshake, no http)
with these arguments:
- our client accepts connections with ver<ver_min!
- connections close with 
  - microsoft: ugly System.IO.IOException in System.Net.Sockets.Socket.Receive
  - facebook: receiving an empty message
  - the rest: received an alert
- gnutls wants client authentiation
- TLS_RSA_WITH_DES_CBC_SHA is not supported by microsoft, lcl, facebook, gnutls
- TLS_RSA_WITH_AES_128_CBC_SHA is not supported by microsoft for SSLv3
*)


type Opt = Vl of ProtocolVersion | Cl of CipherSuites | Sl of string

(* allpossib [[a;b;c];[x,y];[1;2]] = [[a;x;1];[a;x;2];...;[c;y;2]] *)
let rec allpossib = function
    [] -> []
  | [a] -> List.map (fun x -> [x]) a
  | h::t -> let tt = allpossib t in
    List.fold_right (fun x y -> (List.map (fun z -> x::z) tt) @ y) h []


let print_case ver_min ver_max ccs server_id = 
  Printf.printf "\nTrying with: %s (%s, %s, %s)" server_id 
    (string_of_ProtocolVersion ver_min)
    (string_of_ProtocolVersion ver_max)
    (String.concat "\n\t" (List.map (fun x -> string_of_CipherSuite x) ccs));
  print_newline()


let client_test_all () = 
  let vl = List.map (fun x -> Vl (ver2PV x)) ver_list in
  let cl = List.map (fun x -> Cl [x]) ccslist in
  let sl = List.map (fun x -> Sl ("https://" ^ x)) server_list in
  let all = allpossib [vl;vl;cl;sl] in
  Crypto.set_debug false;
  Record.set_debug false;
  Handshake.set_debug false;
  let rec test l =
    match l with
    | [] -> ()
    | [Vl ver_min; Vl ver_max; Cl ccs; Sl server_id]::t ->
	begin
	  try 
	    print_case ver_min ver_max ccs server_id;
	    let net_conn = Net.connect server_id in
	    let connid, sessid = connect_ext net_conn ver_min ver_max ccs server_id in
	    close connid false;
	    test t
	  with
	    e -> 
	      begin 		
		Printf.printf "%s\n" (e.ToString());
		test t
	      end
	end
    | _ -> failwith "my error"
  in
  test all






let rec fork_server f sock () =  
  let conn = Net.accept sock in
  try 
    let  _,_ =  f conn in 
    Pi.fork (fork_server f sock)
  with 
  //| :? System.IO.IOException -> printf "Network problems... connection is closed\n"
  //| :? System.Net.Sockets.SocketException -> printf "Network problems... connection is closed\n"
  | x -> 
      Printf.printf "%s\n" (x.ToString());
      Pi.fork (fork_server f sock)


(* we add a user-password pair in the password database *)
let _ = App_passwd.insertPasswd (Data.str "guest") (Data.str !certname) (utf8 (str "password"))


let _ = 
  //let new_argv = Array.sub Sys.argv 0 2 in
  //Arg.parse current new_argv [] select_entity usage_str;
  if Array.length Sys.argv = 1 then 
    Printf.printf "\nusage: [client args | server args ]\n" 
  else
    if !http && !passwd then
      printf "\nuse -http or -passwd options (or none), but not both\n"
    else
    begin
      select_entity Sys.argv.(1);
      match !entity with
      | Client -> 
	  if !rehandshake_par then (* rehandshake is equivalent with -reconnect and -close *)
	    begin close_notify := true; reconnect := true end
	  else if !passwd then
	    clientPasswd (str !uri)
	  else if !reconnect then
	    client_resume2 ()
//	  else if !scenario then
//	    clientUnlimHttp (str !uri)
	  else if !ctestall then
	    client_test_all ()
	  else if !downgrade then
	    //server_ver_downgrade (ver2PV !ver_max) (ver2PV !ver_min) !uri !http !close_notify !wait_close
	      server_ver_downgrade2 ()
	  else if !http then
	    //clientHttpApp !uri
            let conn = Net.connect (str !uri) in 
//	    let shost = Net.get_host (str !uri) in 
            let connid,sessid = connect conn in
            httpGet connid !uri
	  else
	    begin
	      let conn = Net.connect (str !uri) in 
//	      let shost = Net.get_host (str !uri) in 
	      let connid,sessid = connect conn in
	      if !close_notify then
		ignore (close connid !wait_close)
	    end
      | Server -> 
	  if !passwd then
	    Pi.fork (serverPasswd (str defaultUri) (str !certname))
	  else	    
	    let sock = Net.listen (str defaultUri) in
	    if !http then
	      let f = 
		(fun conn -> 
		  let cid,sid = accept_ext conn !certname !ver_min !ver_max !csl in
		  serverHttpResp conn cid sid;
		  cid,sid)
	      in
	      fork_server f sock ()		
	    else
	      let f = 
		(fun conn -> 
		  accept_ext conn !certname !ver_min !ver_max !csl)
	      in
	      fork_server f sock ()
    end

#else (* if concrete *)

#if recordonly   
(* generate keys statically (no handshake) and do a message exchange *)

let updateConn_net connid netconn = 
  let conn = getConnection connid in
  storeConnection connid 
    { net_conn = netconn;
      entity = conn.entity;
      sessid = empty_bstr;
      crt_version = conn.crt_version;
      read = conn.read;
      write = conn.write; 
    }

let recordonly_example () = 
  let cconnid, sconnid = generateConnection () in
  Pi.fork 
    (fun () -> 
      begin
	let sock = Net.listen (defaultUri) in
	let conn = Net.accept sock in
	updateConn_net sconnid conn;
	let msg = recv sconnid in
	send sconnid msg
      end
    );
  Pi.fork (fun () -> 
    Pi.send seqnumChan empty_bstr; 
    Pi.send seqnumChan empty_bstr;
    Pi.send seqnumChan empty_bstr; 
    Pi.send seqnumChan empty_bstr);
  let conn = Net.connect (defaultUri) in 
  updateConn_net cconnid conn;
  print_string "Please enter a line of text"; print_newline();
  let payload = read_line () in
  send cconnid (utf8 (str payload));
  let msg = recv cconnid in
  print_string ("We have received: "^(istr (iutf8 msg)));
  print_newline()

let _ = recordonly_example ()

#else (* if recordonly *)

(* we add a user-password pair in the password database *)
let _ = App_passwd.insertPasswd (Data.str "guest") (Data.str defaultUri) (utf8 (str "password"))


let none str = raise (Arg.Bad str)

let speclist = [reconnect_arg; rehandshake_arg; passwd_arg]

let _ = Prins.genX509 (Data.str defaultUri)


let client () = 
  let conn = Net.connect (defaultUri) in 
  let _,_ = connect conn (str defaultUri) in
  ()
  
let server () = 
  let sock = Net.listen (defaultUri) in
  let conn = Net.accept sock in
  let  _,_ =  accept conn (str defaultUri) in 
  ()

let client_resume () = 
  let conn = Net.connect (defaultUri) in 
  let _,sessid = connect conn (str defaultUri) in
  //Net.close conn;
  //let conn = Net.connect (defaultUri) in 
  let tlsconn,sess = resume conn sessid in
  ()

let client_rehandshake () = 
  let conn = Net.connect (defaultUri) in 
  let connid,sessid = connect conn (str defaultUri) in
//  Net.close conn;
//  let conn = Net.connect (defaultUri) in 
  let tlsconn,sess = rehandshake connid sessid in
  ()
  
let server_resume () = 
  let sock = Net.listen (defaultUri) in
  let conn = Net.accept sock in
  let  _,_ =  accept conn (str defaultUri) in 
  (* next two lines not really needed; moreover, in fact, Pi.recv hangs when they are used ... *) 
  //Net.close conn;
  //let conn = Net.accept sock in
  let  _,_ =  accept conn (str defaultUri) in 
  ()

let _ = 
  Arg.parse speclist none "usage: command [-reconnect | -passwd]";
  if !passwd && (!reconnect || !rehandshake_par) then
    printf "\n-passwd option is not compatible with -reconnect or -rehandshake\n"
  else if !reconnect then
    begin
      Pi.fork server_resume;
      client_resume ()
    end    
  else if !rehandshake_par then
    begin
      Pi.fork server_resume;
      client_rehandshake ()
    end    
  else if !passwd then
    begin
      Pi.fork (serverPasswd (str defaultUri) (str defaultUri));
      Pi.fork (fun () -> 
	Pi.send seqnumChan empty_bstr; 
	Pi.send seqnumChan empty_bstr;
	Pi.send seqnumChan empty_bstr; 
	Pi.send seqnumChan empty_bstr);
      clientPasswd (Data.str defaultUri)
    end
  else
    begin
      Pi.fork server;
      client ()
    end    

#endif (* if recordonly *)
#endif (* if concrete *)
