(* Copyright (c) Microsoft Corporation.  All rights reserved.  *)
open Printf
open Pi

#if fs
open System.Diagnostics
#endif

type str =
  | Literal of string
  | Base64 of bytes
and 
#if fs
[<DebuggerDisplay("{debugDisplay()}")>] 
#endif
bytes =
  | Concat of bytes * bytes
  | Utf8 of str
  | Fresh of Pi.name
  | Bin of blob
  | Sub of bytes * int
and blob = 
  | Hash of bytes
  | DerivedKey of bytes * bytes    (* PSHA-1(basekey,nonce)/(utf8 pwd,seed) *)
  | SymSign of bytes * bytes
  | SymEncrypt of bytes * bytes
  | PK of bytes // from signing key to verification key
  | AsymSign of bytes * bytes
  | AsymEncrypt of bytes * bytes // pkEncrypt(pke,text)
  | X509Cert of str * str * str * str * bytes // (issuer,serial number,subject,algorithm,publickey)

#if fs
// pretty-printing

let rec syms_str s = match s with
  | Base64(t) -> 1 + (syms_bytes t)
  | Literal(s) -> 1
and syms_blob b = 
  let keyed key arg = 1 + (syms_bytes key) + (syms_bytes arg) in
  match b with
    | SymSign(k,t) -> keyed k t
    | AsymSign(k,t) -> keyed k t
    | AsymEncrypt(k,t) -> keyed k t
    | SymEncrypt(k,t) -> keyed k t
    | PK(s) -> 1 + (syms_bytes s)
    | Hash(s) -> 1 + (syms_bytes s)
    | DerivedKey(k,t) -> keyed k t
    | X509Cert(i,n,u,a,p) -> 3 + (syms_str u) + (syms_str a) + (syms_bytes p)
and syms_bytes b = 
  match b with
    | Utf8(s) -> 1 + (syms_str s)
    | Concat(x,y) -> 1 + (syms_bytes x) + (syms_bytes y)
    | Fresh(n) -> 1
    | Bin(b) -> syms_blob b
    | Sub(x,i) -> syms_bytes x

let lastHash (u:string) = 
  let len = String.length u in
    if String.contains u '#' then
      let i = String.rindex u '#' in
	if i+1 < len
	then (String.sub u (i+1) (len-i-1))
	else u
    else u

let lastUri (u:string) = 
  let nu = lastHash u in
  let len = String.length nu in
    if String.contains nu '/' then
      let i = String.rindex nu '/' in
	if i+1 < len then
	  (String.sub nu (i+1) (len-i-1))
	else nu
    else nu

let indent = ref 0
let spaces () = String.make (!indent) ' ' 

let rec fpstr o s = match s with
  | Base64(t) -> fprintf o "%a" fpbytes t ; 
  | Literal(s) -> fprintf o "%s" (lastUri s)

and fpblob o b = 
  let keyed name key arg = fprintf o "%s{%a}[%a]" name fpbytes key fpbytes arg in
  match b with
    | SymSign(k,t) -> keyed "HMAC-SHA1" k t; 
    | AsymSign(k,t) -> keyed "RSA-SHA1" k t; 
    | AsymEncrypt(k,t) -> keyed "RSA-Enc" k t; 
    | SymEncrypt(k,t) -> keyed "AES-Enc" k t; 
    | PK(s) -> fprintf o "PK(%a)" fpbytes s ; 
    | Hash(s) -> fprintf o "SHA1(%a)" fpbytes s ; 
    | DerivedKey(k,t) -> keyed "P-SHA1" k t; 
    | X509Cert(i,n,u,a,p) -> fprintf o "X509(%a,%a,%a,%a)" fpstr i fpstr u fpstr a fpbytes p; 
and fpbytes o b = 
  match b with
    | Utf8(s) -> fprintf o "%a" fpstr s; 
    | Concat(x,y) -> fprintf o "%a | %a" fpbytes x fpbytes y; 
    | Fresh(n) -> fprintf o "%a" Pi.fpname n; 
    | Bin(b) -> fprintf o "%a" fpblob b;
    | Sub(x,i) -> fprintf o "%a %d" fpbytes x i

let rec spstr o s = match s with
  | Base64(t) -> sprintf "%a" spbytes t 
  | Literal(s) -> sprintf "%s" s
and spblob o b = 
  let keyed name key arg = sprintf "%s{%a}[%a]" name spbytes key spbytes arg in
  match b with
    | SymSign(k,t) -> keyed "HMAC-SHA1" k t
    | AsymSign(k,t) -> keyed "RSA-SHA1" k t
    | AsymEncrypt(k,t) -> keyed "RSA-Enc" k t
    | SymEncrypt(k,t) -> keyed "AES-Enc" k t
    | PK(s) -> sprintf  "PK(%a)" spbytes s 
    | Hash(s) -> sprintf "SHA1(%a)" spbytes s
    | DerivedKey(k,t) -> keyed "P-SHA1" k t
    | X509Cert(i,n,u,a,p) -> sprintf "X509(%a,%a,%a,%a)" spstr i spstr u spstr a spbytes p
and spbytes o b = 
  match b with
    | Utf8(s) -> sprintf  "\'%a\'" spstr s
    | Concat(x,y) -> sprintf "%a | %a" spbytes x spbytes y
    | Fresh(n) -> sprintf "Fresh[]" 
    | Bin(b) -> sprintf "%a" spblob b
    | Sub(x,i) -> sprintf "%a %d" spbytes x i

type bytes with
  member x.debugDisplay() = sprintf "bytes: %a" spbytes x (* or other display form *)
end
#endif

let str s = Literal s
let istr s = match s with
  | Literal s -> s
  | _ -> failwith "istr failed"

let bytes b = Fresh b
let ibytes b = match b with
    Fresh(b) -> b
  | _ -> failwith "ibytes failed"

let base64 b = Base64(b)
let ibase64 b = match b with
| Base64 s -> s
| _ -> failwith "ibase64 failed"

let concat x y = Concat(x,y)
let iconcat s = match s with
  | Concat(x,y) -> (x,y)
  | _ -> failwith "iconcat failed"


let concat3 x y z = concat x (concat y z)
let concat4 x y z w = concat x (concat3 y z w)
let concat5 x y z w v  = concat x (concat4 y z w v)
let concat6 x y z w v u = concat x (concat5 y z w v u)


let concatvar x y = concat x y
let iconcat3 e = (let x,yz = iconcat e in let y,z = iconcat yz in x,y,z)
let iconcat4 e = (let x,yzw = iconcat e in let y,zw = iconcat yzw in let z,w = iconcat zw in x,y,z,w)
let iconcat5 e = (let x,yzwv = iconcat e in let y,zwv = iconcat yzwv in let z,wv = iconcat zwv in let w,v = iconcat wv in x,y,z,w,v)
let iconcat6 e = (let x,yzwvu = iconcat e in let y,zwvu = iconcat yzwvu in let z,wvu = iconcat zwvu in let w,vu = iconcat wvu in let v,u = iconcat vu in x,y,z,w,v,u)




let length (x:bytes) = 0

let utf8 x = Utf8 x
let iutf8 b = match b with
| Utf8 s -> s
| _ -> failwith "iutf8 failed"

let guid () = Base64(Fresh(Pi.name "nonce"))

#if fs
let sprint_raw msg = spbytes () msg
let fprint_raw out msg = fpbytes out msg
let print_raw msg = fpbytes stdout msg
let print_bytes s msg = Printf.printf "\n%s\n" s; print_raw msg
let fprint_bytes out s msg = Printf.fprintf out "\n%s\n" s; fpbytes out msg
#endif



(* conversions between bytes and "basic" types: int, etc *)

let bytes_of_int (nb:int) (i:int) = Utf8 (str "int")

let int_of_bytes (nb:int) x = 
  match x with
  | Utf8 s -> 0
  | _ -> failwith "wrong int represenatation" 


let bytes_of_intpair ((i,j):int*int) = Utf8 (str "int,int")

let intpair_of_bytes x = 
  match x with
  | Utf8 s -> 0,0
  | _ -> failwith "wrong intpair represenatation"
