(* Copyright (c) Microsoft Corporation.  All rights reserved.  *)
open Data

type nonce = bytes
type seed = bytes
type guid = bytes

type key = 
    SymKey of bytes 
  | MsymKey of bytes 
  | AsymPrivKey of bytes
  | AsymPubKey of bytes

type mkey = key 

type keyseed = bytes

let keytobytes kb = match kb with
    SymKey(k) -> k
  | MsymKey(k) -> k
  | AsymPrivKey(k) -> k
  | AsymPubKey(k) -> k


type macs = Mac of bytes
type blocks = Blocks of bytes

let symkey k = SymKey k
let msymkey k = MsymKey k

(* *)

let asympubkey k = AsymPubKey k
let asymprivkey k = AsymPrivKey k

let asympub k = AsymPubKey (utf8 k)
let asympriv k = AsymPrivKey (utf8 k)

let freshbytes s = Fresh(Pi.name s)
let mkKey() = freshbytes "key"
let mkKey256() = freshbytes "key"

// Nonce generation

let rand_nonce () = freshbytes "nonce"
let rand_seed () = freshbytes "seed"
let rand_guid () = freshbytes "guid"

let mkNonce () = rand_nonce()
let mkNonce256 () = rand_nonce()
let mkRandom (n:int) = freshbytes "nonce"


let secret_nonce (x:nonce) = ()
let secret_bitstring (x:bytes) = ()

// Timestamps

// let time () = (base64 (freshbytes "now"), base64 (freshbytes "plusone"))

let now () = (base64(Fresh (Pi.name "Now")),str "PlusOneMinute")

// Hashing

let sha1 x = Bin(Hash(x))
let md5 x = sha1 x
let sha256 x = sha1 x


//  (SSL/TLS-like) Derived Key generation 

let prf k s rnd (l:int) = Bin(DerivedKey(k,Concat(s,rnd)))
let prf_hmac k s rnd l = prf k s rnd l
let checkprf_hmac k s rnd (l:int) f = Bin(DerivedKey(k,Concat(s,rnd))) = f
let ssl_prf k rnd (l:int) = Bin(DerivedKey(k,rnd))

// Symmetric encryption

let sym_encrypt key (Blocks text) = Blocks(Bin(SymEncrypt(key,text)))
let sym_decrypt key msg = 
  match msg with
  | Blocks(Bin(SymEncrypt(key',text))) when key = key' -> Blocks text
  | _ -> failwith "sym_decrypt failed"

// Asymmetric encryption

(* for CV only: only the interfaces matters *)
let dec (k:key) (c:bytes) = c
let enc (k:key) (m:bytes) (iv:bytes) = m
let deca (k:key) (c:bytes) = c
let enca (k:key) (m:bytes) (s:bytes) = m
let pkgen (seed:bytes) = SymKey seed
(* *)

let asym_encrypt key text = Bin(AsymEncrypt(key,text))
let asym_decrypt key msg = 
  match msg with
  | Bin(AsymEncrypt(Bin(PK(key')),text)) when key = key' -> text
  | _ -> failwith "asym_decrypt failed"


// Symmetric signature

let sym_sign key text = Bin(SymSign(key,text))

// Asymmetric signature

let asym_sign key msg = Bin(AsymSign(key,msg))
let asym_signature_verify key msg hash = match (key,hash) with
    (Bin(PK(vk)),Bin(AsymSign(sk,m))) when sk = vk -> if m = msg then ()
    else failwith "asym_signature_verify: signed values do not match"
  | _ -> failwith "asym_signature_verify failed"
  

// AES encryption

let aes_encrypt k text = 
  match k with 
      SymKey key -> sym_encrypt key text
    | _ -> failwith "cannot use aes_encrypt with asymmetric key"

let aes_decrypt k (msg:blocks) = 
  match k with 
      SymKey key -> sym_decrypt key msg
    | _ -> failwith "cannot use aes_decrypt with asymmetric key"

let aes_encrypt_wiv k (iv:bytes) p = aes_encrypt k p
let aes_decrypt_wiv k (iv:bytes) p = aes_decrypt k p

// DES encryption

let des_decrypt k (iv:bytes) p = aes_decrypt_wiv k (iv:bytes) p
let des_encrypt k (iv:bytes) p = aes_encrypt_wiv k (iv:bytes) p


// RSA encryption 

let rsa_pub k = 
  match k with 
      AsymPrivKey s -> AsymPubKey(Bin(PK s))
    | _ -> failwith "cannot use rsa_pub with symmetric key"

let rsa_keygen () = AsymPrivKey (freshbytes "rsa_secret")

let rsa_encrypt key t = 
  match key with 
      AsymPubKey k -> asym_encrypt k t
    | _ -> failwith "cannot use rsa_encrypt with symmetric or private key"

let rsa_decrypt key e = 
  match key with 
      AsymPrivKey k -> asym_decrypt k e
    | _ -> failwith "cannot use rsa_decrypt with symmetric or public key"

let rsa_encrypt_oaep key t = 
  match key with 
      AsymPubKey k -> asym_encrypt k t
    | _ -> failwith "cannot use rsa_encrypt with symmetric or private key"

let rsa_decrypt_oaep key e = 
  match key with 
      AsymPrivKey k -> asym_decrypt k e
    | _ -> failwith "cannot use rsa_decrypt with symmetric or public key"

// RSA-SHA1

let rsa_sign key t = 
  match key with 
      AsymPrivKey k -> asym_sign k t
    | _ -> failwith "cannot use rsa_sign with symmetric or public key"

let rsa_verify key t d = 
  match key with 
      AsymPubKey k -> asym_signature_verify k t d
    | _ -> failwith "cannot use rsa_verify with symmetric or private key"



// HMAC-SHA1

let noncetobytes (n:nonce) = n
let keyseedtobytes (n:bytes) = n
let bytestokeyseed (n:bytes) = n
let bytestononce (n:bytes) = n
let mactobytes (Mac n) = n
let bytestomac n = Mac n
let blockstobytes (Blocks n) = n
let bytestoblocks n = Blocks n

let hmacmd5 k m = 
  match k with 
      MsymKey key -> Mac (sym_sign key m)
    | _ -> failwith "cannot use hmacmd5 with asymmetric key"
//Bin(DerivedKey(keytobytes k,m))

let hmacsha1 k text = 
  match k with 
      MsymKey key -> Mac (sym_sign key text)
    | _ -> failwith "cannot use hmacsha1 with asymmetric key"

let hmacmd5_verify k v  m =  
  (hmacmd5 k v) = m

let hmacsha1_verify k v  m =  
  (hmacsha1 k v) = m

// X509

let sha1RSA = str "sha1RSA" 

let x509signcert i s key = 
  let n = guid() in
    Bin(X509Cert(i,n,s,sha1RSA,key))

let x509sign key t = 
  match key with 
      AsymPrivKey k -> asym_sign k t
    | _ -> failwith "cannot use x509sign with symmetric or public key"

let x509pubkey s = match s with
| Bin(X509Cert(_,_,_,_,k)) -> k 
| _ -> failwith "not an X509 cert"

let x509subject s = match s with
| Bin(X509Cert(_,_,s,_,_)) -> s
| _ -> failwith "not an X509 cert"

let x509number s = match s with
| Bin(X509Cert(_,n,_,_,_)) -> n
| _ -> failwith "not an X509 cert"

let x509issuer s = match s with
| Bin(X509Cert(i,_,_,_,_)) -> i
| _ -> failwith "not an X509 cert"

let x509alg s = match s with
| Bin(X509Cert(_,_,_,a,_)) -> a
| _ -> failwith "not an X509 cert"


let x509verify (cert:bytes) (subject:str) : bool = 
  failwith "not in the symbolic mode"

let x509analyse (cert:bytes) : bool =
  failwith "not in the symbolic mode"

let getCN (x:str) = x


// ARCFOUR
type arcfour_state = key 
type stream_cipher_state = ArcFourState of arcfour_state

let stream_cipher_init (k:bytes) = 
  ArcFourState (SymKey k)

let stream_cipher_encrypt (p:bytes) (st:stream_cipher_state) = 
  match st with
  | ArcFourState (SymKey k) -> blockstobytes(sym_encrypt k (bytestoblocks p))
  | _ -> failwith "invalid stream cipher state"





