open Data
open Formats_ciphersuites
open Formats


let zeros (nb:int) = Utf8 (str "zeros")
let bytes_of_seq (cseq:int) = Utf8 (str "seq:int")


(* constants *)

let empty_bstr = Utf8 (str "empty bitstring")
let empty_sid = Utf8 (str "empty sid")
let ccs_msg = Utf8 (str "CCS")

(* SSLv3 constants *)
 
let ssl_pad1_md5 = Utf8 (str "ssl_pad1_md5")
let ssl_pad2_md5 = Utf8 (str "ssl_pad2_md5")
let ssl_pad1_sha1 = Utf8 (str "ssl_pad1_sha1")
let ssl_pad2_sha1 = Utf8 (str "ssl_pad2_sha1")

let ssl_sender_client = Utf8 (str "ssl_sender_client")
let ssl_sender_server = Utf8 (str "ssl_sender_server")


(* versions *)

let version_of_bytes (msg: bytes) =  msg
let bytes_of_version (ver:version) = ver

let ver2PV (ver:version):ProtocolVersion = ver
let PV2ver (pv:ProtocolVersion):version = pv

let PV2pmver (pv:ProtocolVersion) = 
  match pv with
//  | Utf8 (Literal "SSLv2") ->  PM_SSLv2
//  | Utf8 (Literal "SSLv3") ->  PM_SSLv3
  | Utf8 (Literal "TLS1p0") ->  PM_TLS1p0
  | Utf8 (Literal "TLS1p1") ->  PM_TLS1p1
//  | Utf8 (Literal "TLS1p2") ->  PM_TLS1p2
  | _ -> failwith "internal error"


(* content types *)

let ContentType_of_bytes (msg: bytes) =  msg
let bytes_of_ContentType (ct:ContentType) = ct


(* alerts *)

let Alert_of_bytes x = 
  match x with
  | Utf8 (Literal "alert") ->  {level = AL_fatal; description = AD_close_notify }
  | _ -> failwith "only one alert"

let bytes_of_Alert (x:Alert) = Utf8 (Literal "alert")

let bytes_of_AlertDescription (x:AlertDescription) = Utf8 (Literal "alert_desc")

let string_of_Alert (a:Alert) = "alert"


(* handshake types *)

let bytes_of_HandshakeType (ht: HandshakeType) = ht
let HandshakeType_of_bytes (msg: bytes) = msg


(* ciphersuites *)

let CipherSuite_of_bytes x = 
  match x with
  | Utf8 (Literal "TLS_NULL_WITH_NULL_NULL")      -> TLS_NULL_WITH_NULL_NULL
  | Utf8 (Literal "TLS_RSA_WITH_AES_128_CBC_SHA") -> TLS_RSA_WITH_AES_128_CBC_SHA
  | _ -> failwith "wrong ciphersuite" 
	
let bytes_of_CipherSuite x = 
  match x with
  | TLS_NULL_WITH_NULL_NULL -> Utf8 (Literal "TLS_NULL_WITH_NULL_NULL")
  | TLS_RSA_WITH_AES_128_CBC_SHA -> Utf8 (Literal "TLS_RSA_WITH_AES_128_CBC_SHA")


let CipherSuites_of_bytes (msg: bytes) = 
  iconcat msg

let bytes_of_CipherSuites (csl: CipherSuites) = 
  let (x,y) = csl in
  concat x y


(* compression methods *)

let cm_null = Utf8 (str "CM_null")

let CompressionMethod_of_bytes (msg: bytes) : CompressionMethod = 
  if msg = cm_null then
    CM_null
  else
    failwith "wrong compression method" 

let bytes_of_CompressionMethod (cm:CompressionMethod) : bytes = 
  match cm with
  | CM_null -> cm_null


let bytes_of_CompressionMethods (cml:CompressionMethods) : bytes = 
  bytes_of_CompressionMethod cml

let CompressionMethods_of_bytes (msg: bytes) : CompressionMethods =
  CompressionMethod_of_bytes msg



(* parsing functions *)

let parseClientHello (msg: bytes) = iconcat6 msg
let parseServerHello (msg: bytes) = iconcat5 msg
let parsePMS (msg: bytes) = iconcat msg
let parseHandshake (msg: bytes) = iconcat3 msg

let parseRecord (msg:bytes) = 
  let header,fragment = iconcat msg in
  let ct,ver,len = iconcat3 header in
  (ct, ver, len, fragment)

let get_last_block (bs:int) (msg:bytes) = empty_bstr




(* other functions *)
let selectVersion (clver:ProtocolVersion) (svermin:ProtocolVersion) (svermax:ProtocolVersion) = 
  clver

let selectCipherSuite (csl:CipherSuites) (pref_csl:CipherSuites) = 
  let cs1, cs2 = csl in
  match CipherSuite_of_bytes cs1 with
  | TLS_RSA_WITH_AES_128_CBC_SHA -> cs1
  | TLS_NULL_WITH_NULL_NULL -> failwith "handshake failure"

let selectCompressionMethod cml = 
  if cml = CM_null then
    CM_null
  else
    failwith "handshake failure"

let lt_ver (ver1:ProtocolVersion) (ver2:ProtocolVersion) : bool = 
  failwith "not available symbolically"
