(* Copyright (c) Microsoft Corporation.  All rights reserved.  *)


#if fs

(* Data types used in the concrete implementation *)
#if concrete

type bytes = byte[] 
type str = string

(* Data types used in the symbolic implementation *)
#else

type str =
  | Literal of string
  | Base64 of bytes

and bytes =
  | Concat of bytes * bytes
  | Utf8 of str
  | Fresh of Pi.name
  | Bin of blob
  | Sub of bytes * int
  
and blob = 
  | Hash of bytes
  | DerivedKey of bytes * bytes 
  | SymSign of bytes * bytes
  | SymEncrypt of bytes * bytes
  | PK of bytes 
  | AsymSign of bytes * bytes
  | AsymEncrypt of bytes * bytes
  | X509Cert of str * str * str * str * bytes

#endif

(* Data types exposed to the opponent *)
#else
type blob
type str =
  | Literal of string
  | Base64 of bytes
and bytes = 
  | Concat of bytes * bytes
  | Utf8 of str
  | Fresh of Pi.name
  | Bin of blob
  | Sub of bytes * int
#endif

val str: string -> str
val istr: str -> string

val base64: bytes -> str
val ibase64: str -> bytes

val utf8: str -> bytes              
val iutf8: bytes -> str             

(* length insensitive "raw" concatenation *)
val concat: bytes -> bytes -> bytes
val concat3: bytes -> bytes -> bytes -> bytes
val concat4: bytes -> bytes -> bytes -> bytes -> bytes
val concat5: bytes -> bytes -> bytes -> bytes -> bytes -> bytes
val concat6: bytes -> bytes -> bytes -> bytes -> bytes -> bytes -> bytes

(* length sensitive concatenation *)
val concatvar: bytes -> bytes -> bytes  
val iconcat: bytes -> bytes * bytes  
val iconcat3: bytes -> (bytes * bytes * bytes)
val iconcat4: bytes -> (bytes * bytes * bytes * bytes)
val iconcat5: bytes -> (bytes * bytes * bytes * bytes * bytes)
val iconcat6: bytes -> (bytes * bytes * bytes * bytes * bytes * bytes)





val length: bytes -> int

#if concrete
val rev:bytes -> bytes
#endif

val guid:unit -> str

(* Pretty-printing functions *)
#if fs 
val fpstr: out_channel -> str -> unit 
val fpbytes: out_channel -> bytes -> unit 
val syms_str: str -> int
val syms_bytes: bytes -> int

val sprint_raw : bytes -> string
val fprint_raw : out_channel -> bytes -> unit
val print_raw : bytes -> unit
val fprint_bytes : out_channel -> string -> bytes -> unit
val print_bytes : string -> bytes -> unit
#endif


val bytes_of_int: int -> int -> bytes
val int_of_bytes: int -> bytes -> int

val intpair_of_bytes: bytes -> (int*int)
val bytes_of_intpair: (int*int) -> bytes
