(* Copyright (c) Microsoft Corporation.  All rights reserved.  *)
open Data 


type key
type mkey = key
type keyseed = bytes
type nonce = bytes
type seed = bytes
type guid = bytes

#if concrete
type macs = bytes
type blocks = bytes
#else
type macs
type blocks
#endif


val mkNonce: unit -> nonce // Generate a fresh nonce on 16 bytes
val mkNonce256: unit -> nonce // Generate a fresh nonce on 32 bytes

val rand_nonce: unit -> nonce // Generate a fresh nonce
val rand_guid : unit -> guid
val rand_seed: unit -> seed // Generate a fresh nonce

val secret_nonce: nonce -> unit // Queries for the secrecy of the arguement
val secret_bitstring: bytes -> unit // Queries for the secrecy of the arguement

val keytobytes: key -> bytes
val noncetobytes : nonce -> bytes
val keyseedtobytes : keyseed -> bytes
val bytestokeyseed : bytes -> keyseed
val bytestononce : bytes -> nonce
val mactobytes : macs ->bytes
val bytestomac : bytes -> macs
val blockstobytes : blocks ->bytes
val bytestoblocks : bytes -> blocks


val symkey: nonce -> key
val msymkey: nonce -> mkey

val mkRandom: int -> bytes
val mkKey: unit -> bytes
val mkKey256: unit -> bytes

val sha256: bytes -> bytes

val aes_encrypt_wiv: key -> bytes -> blocks -> blocks // aes_encrypt_wiv(key,iv,plaintext)
val aes_decrypt_wiv: key -> bytes -> blocks -> blocks // aes_decrypt_wiv(key,iv,ciphertext)
val des_encrypt: key -> bytes -> blocks -> blocks // des_encrypt(key,iv,plaintext)
val des_decrypt: key -> bytes -> blocks -> blocks // des_decrypt(key,iv,ciphertext)

#if concrete
type arcfour_state
#else
type arcfour_state = key
#endif

type stream_cipher_state = ArcFourState of arcfour_state
val stream_cipher_init: bytes -> stream_cipher_state 
val stream_cipher_encrypt: bytes -> stream_cipher_state -> bytes


val asympubkey: bytes -> key
val asymprivkey: bytes -> key
val asympub: str -> key
val asympriv: str -> key

val x509signcert: str -> str -> bytes -> bytes
val x509issuer: bytes -> str
val x509number: bytes -> str
val x509subject: bytes -> str
val x509alg: bytes -> str
val x509pubkey: bytes -> bytes
val x509analyse: bytes -> bool
val x509verify: bytes -> str -> bool
val getCN: str -> str

(* for CV *)
val enc : key -> bytes -> bytes -> bytes
val dec : key -> bytes -> bytes
val enca : key -> bytes -> bytes -> bytes
val deca : key -> bytes -> bytes
val pkgen : bytes -> key
(* *)

val rsa_pub: key -> key
val rsa_keygen: unit -> key
val rsa_encrypt: key -> bytes -> bytes
val rsa_decrypt: key -> bytes -> bytes

val rsa_encrypt_oaep: key -> bytes -> bytes
val rsa_decrypt_oaep: key -> bytes -> bytes

val rsa_sign: key -> bytes -> bytes
val rsa_verify: key -> bytes -> bytes -> unit

val ssl_prf: bytes -> bytes -> int -> bytes


val md5: bytes -> bytes
val sha1: bytes -> bytes
val aes_encrypt: key -> blocks -> blocks 
val aes_decrypt: key -> blocks -> blocks
val prf: bytes -> bytes -> bytes -> int -> nonce
val prf_hmac: bytes -> bytes -> bytes -> int -> bytes
val checkprf_hmac: bytes -> bytes -> bytes -> int -> bytes -> bool
val hmacmd5: mkey -> bytes -> macs
val hmacmd5_verify: mkey -> bytes -> macs -> bool
val hmacsha1: mkey -> bytes -> macs
val hmacsha1_verify: mkey -> bytes -> macs -> bool


#if paper
(*--- cryptoBegin *)
module Crypto
  val mkNonce: unit -> bytes   // Generate a fresh nonce
  val md5: bytes -> bytes      // MD5
  val sha1: bytes -> bytes     // SHA1
  val aes_encrypt: symkey -> bytes -> bytes    // AES Encryption
  val aes_decrypt: symkey -> bytes -> bytes 
  val hmacsha1: mackey -> bytes -> bytes       // HMACSHA1
  val hmacsha1Verify: mackey -> bytes -> bytes -> bool
  val rsa_encrypt: pubkey -> bytes -> bytes    // RSA Encryption
  val rsa_decrypt: privkey -> bytes -> bytes
(*--- cryptoEnd *)
#endif


#if concrete
val set_debug: bool -> unit
#endif
