(* Copyright (c) Microsoft Corporation.  All rights reserved.  *)
(********************************************************)
(* The Database of Principals *)
(*********************************************************
   Defines:
     type principal: Principals
 *********************************************************)

#light "off"

module Prins

open Pi
open Data
open Crypto

(* Users and Passwords *)

type principalU = 
      {user:str; 
       password: str;}

let principalU (user:str) (password:str) =  {user=user;password=password}
let user pu = pu.user
let password pu = pu.password

let pwdDb = Db.create ()

let genUserPassword (u:str) : unit = failwith "Disabled in concrete code. Usernames and passwords must be set up out of band"

let regUserPassword u p = Db.insert pwdDb u (principalU u p)

let getUserPassword u = Db.select pwdDb u

let getPassword (u:str) = 
  let p = getUserPassword u in
    p.password

(* X.509 Certificates *)

type principalX = 
       {subject:str;
	cert: bytes;
	pubkey: key;
	privkey: key;}


let principalX (subject:str) (cert:bytes) (pubkey:key) (privkey:key) = 
{subject=subject;cert=cert;pubkey=pubkey;privkey=privkey}
let subject px = px.subject
let cert px = px.cert
let pubkey px = px.pubkey
let privkey px = px.privkey

let genX509 (u:str) : unit = failwith "Disabled in concrete code. Use makecert.exe or acquire and install a signed certificate."

open System.Security.Cryptography.X509Certificates
open System.Security.Cryptography
let localMachine:X509Store = 
  new X509Store(StoreName.My,StoreLocation.LocalMachine) 
let currentUser:X509Store = 
  new X509Store(StoreName.My,StoreLocation.CurrentUser) 
exception Cert_Not_Found of string

let getX509Certificate (store:X509Store) (subject:string) :  X509Certificate2 = 
  store.Open(OpenFlags.ReadOnly);
  let certs = store.get_Certificates() in
  let search = certs.Find(X509FindType.FindBySubjectName,subject,false) in
  if (search.get_Count()) < 1 then raise (Cert_Not_Found subject )
  else  
    let cert = search.Item(0) in
      cert

let getX509Prin (store:X509Store) (subject:string) : principalX =
  let cert = getX509Certificate store subject in
    if (cert.get_HasPrivateKey()) = false then raise (Cert_Not_Found subject )
    else  
      let priv = (cert.get_PrivateKey()).ToXmlString(true) in
	principalX subject (cert.Export(X509ContentType.Cert))
	  (asympub priv)
	  (asympriv priv)

let getX509Cert (subject:string) : bytes = 
  let c = try (getX509Certificate currentUser subject) with 
           _ -> getX509Certificate localMachine subject in
    c.Export(X509ContentType.Cert)

let getX509 (subj:str) = 
  try (getX509Prin currentUser subj)
  with _ -> getX509Prin localMachine subj

let getPublicKey (u:str) = 
  let c = getX509Cert u in
    asympubkey (x509pubkey c)

let getPrivateKey (u:str) = 
  let p = getX509 u in
    p.privkey

(* Compromizing Principals *)

let leakEv = Pi.trace()

let leakX509 (u:str) = 
  let prin = getX509 u in
    Pi.log leakEv u;
    prin

let leakUserPassword (u:str) = 
  let prin = getUserPassword u in
    Pi.log leakEv u;
    prin







