(* Copyright (c) Microsoft Corporation.  All rights reserved.  *)

#light "off"

module Net

open Data
open System.Net
open Printf

type conn = Sock of Sockets.NetworkStream
type listener = Listener of Sockets.TcpListener

let fake (): conn = 
  failwith "not for concrete executions"

let connect u = 
  let uri = new System.Uri(u) in
  let h = uri.Host in
  let p = uri.Port in
  Printf.printf "Connecting to %s:%d\n" h p;
  let c = new Sockets.TcpClient(h,p) in
  let netstream = c.GetStream() in	
  //netstream.set_ReadTimeout(1000);//1 sec
  //Printf.printf "Timeout is %d\n" (netstream.get_ReadTimeout());
    Sock netstream

let listen u =
  let uri = new System.Uri(u) in
//  let h = Dns.GetHostEntry("localhost").AddressList.[0] in
  let h = IPAddress.Parse(uri.Host) in
  let p = uri.Port in
  Printf.printf "Listening at %s:%d\n" (h.ToString()) p;
  let sock = new Sockets.TcpListener (h,p) in
    sock.Start (); Listener sock
   
let accept (Listener sock) =
  Printf.printf "\nAccepting...\n";
  let tcp = sock.AcceptTcpClient () in
  let netstream = tcp.GetStream() in	
  //netstream.set_ReadTimeout(1000);//1 sec
  //Printf.printf "Timeout is %d\n" (netstream.get_ReadTimeout());
    Sock netstream

let close (Sock s) = s.Close ()

let base64 (x:byte[]) : string = System.Convert.ToBase64String x
let ibase64 (x:string) : byte[] = System.Convert.FromBase64String x

let utf8 (x:string) : byte[] = System.Text.Encoding.UTF8.GetBytes x
let iutf8 (x:byte[]) : string = System.Text.Encoding.UTF8.GetString x

let size (x:byte[]) = 
  let l = Array.length x in 
  [| Byte.of_int(l % 256) ; Byte.of_int(l / 256) |] : byte[]

let sendraw (Sock t) (m:bytes) = 
//  Printf.printf  "Sending: %a\n" fpbytes m;
//  Printf.printf  " containing %d bytes\n" (m.Length);
    t.Write(m,0,m.Length)

let send  (Sock t) (m:bytes) = 
  let s = size m in
    t.Write(s,0,s.Length);
    sendraw (Sock t) m

let sendl  (Sock t) (m:bytes) = 
    sendraw (Sock t) m

let rec read (t:Sockets.NetworkStream) (b:byte[]) (o:int) (l:int) = 
  let r = t.Read(b,o,l) in
    if r < l then
      read t b (o+r) (l-r)
    else b


let recv (Sock t) = 
  let lbuf:byte[] = Bytearray.make 2 in   
  let l = t.Read(lbuf,0,2) in
  let len = Byte.to_int(lbuf.(0))+256*Byte.to_int(lbuf.(1)) in
  let buf = Bytearray.make len in
    (read t buf 0 len)


let recvl (Sock t) = 
  (* it was 4096 *)
  let lbuf:byte[] = Bytearray.make 18438 in   
     let l = t.Read(lbuf,0,18438) in 
     (Bytearray.sub lbuf 0 l)
(* 18437 = 2^14+2048; see rfc2246 sec.6.2(.3) *)


let soapns = "http://schemas.xmlsoap.org/soap/envelope/"
 
// a basic HTTP transport for SOAP messages
// there is no abstract implementation for this module

(* for the first example, it would be nicer to use a plain http post
   with no soap header or response, or even tcp. This matters only
   if we provide all the code for that example. *)

let rec read_length (sr:System.IO.StreamReader) : int =
  let s = sr.ReadLine () in
  if s.StartsWith "Content-Length: " then
    let i = String.length "Content-Length: " in
    int_of_string (s.Substring (i, (s.Length - i)))
  else read_length sr

let http_recv (s:System.IO.Stream) :string =
  let sr = new System.IO.StreamReader (s) in
  let len = read_length sr in
  while (sr.ReadLine () <> "") do () done;
  let buf = Array.create len ' ' in  
  let len2 = sr.ReadBlock (buf, 0, len) in
  assert (len = len2);
  let s = new System.String(buf) in
  s

let http_send (s:System.IO.Stream) (txt:string) : unit =
  Printf.printf  "  Sending: %a\n" fpstr txt;
  Printf.printf  "  containing %i symbols\n" (syms_str txt);
  let sw = new System.IO.StreamWriter(s) in
  sw.Write "HTTP/1.1 200 OK\r\n\r\n";
  sw.Write txt;
  sw.Close()
  
let http_req url action len = 
  Printf.printf  "Requesting url: %s action: %s \n" url action;
  let req = System.Net.WebRequest.Create url in
    req.Method <- "POST";
    req.ContentLength <- Int64.of_int len;
    req.ContentType <- "text/xml; charset=utf-8";
    req.Headers.Add ("SOAPAction", action);
    (req,req.GetRequestStream ())

let writeFile  (f:string) (txt:string) = 
  let sw = new System.IO.StreamWriter(f,true) in 
    sw.Write txt;
    sw.Close()
    


let get_host u = 
  let uri = new System.Uri(u) in
  uri.Host

let get_abspath u = 
  let uri = new System.Uri(u) in
  uri.AbsolutePath
