(* Copyright (c) Microsoft Corporation.  All rights reserved.  *)

#light "off"

open Printf
open System.Security.Cryptography

type bytes = byte[]
type blob = byte[]
type str = string

let str (x:str) = x
let istr (x:str) = x

let bytes (x:byte[]) = x
let ibytes (x:bytes) = x


(*---- Representations. *)

let base64 (x:bytes) : string = System.Convert.ToBase64String x
let ibase64 (x:string) : byte[] = System.Convert.FromBase64String x

let utf8 (x:string) : bytes = System.Text.Encoding.UTF8.GetBytes x
let iutf8 (x:bytes) : string = System.Text.Encoding.UTF8.GetString x

let concatfix (x:bytes) (y:bytes) : bytes = Array.append x y
let iconcatfix (n:int)  (x:bytes) = 
  let x1 = Array.sub x 0 n in 
  let x2 = Array.sub x n (Array.length x - n) in (x1,x2) 

let size (x:bytes) = 
  let l = Array.length x in 
  [| Byte.of_int(l % 256) ; Byte.of_int(l / 256) |] : bytes

let concatvar (x:bytes) (y:bytes) : bytes = concatfix (size x) (concatfix x y)
let iconcatvar (x:bytes) = 
  let s,x' = iconcatfix 2 x in
  let l = Byte.to_int(s.(0))+256*Byte.to_int(s.(1)) in
  iconcatfix l x'


let concat x y = concatfix x y
let concat3 x y z = concat x (concat y z)
let concat4 x y z w = concat x (concat3 y z w)
let concat5 x y z w v  = concat x (concat4 y z w v)
let concat6 x y z w v u = concat x (concat5 y z w v u)


let iconcat z = iconcatvar z
let iconcat3 e = (let x,yz = iconcat e in let y,z = iconcat yz in x,y,z)
let iconcat4 e = (let x,yzw = iconcat e in let y,zw = iconcat yzw in let z,w = iconcat zw in x,y,z,w)
let iconcat5 e = (let x,yzwv = iconcat e in let y,zwv = iconcat yzwv in let z,wv = iconcat zwv in let w,v = iconcat wv in x,y,z,w,v)
let iconcat6 e = (let x,yzwvu = iconcat e in let y,zwvu = iconcat yzwvu in let z,wvu = iconcat zwvu in let w,vu = iconcat wvu in let v,u = iconcat vu in x,y,z,w,v,u)



let rev (x:bytes) : bytes = let y = Array.copy x in System.Array.Reverse y; y

// Pretty printing

let syms_str (s:str) = 0
let syms_bytes (s:bytes) = 0

let fpstr o s = fprintf o "%s" s
let fpbytes o (t:bytes) = 
  let large = 24 in
  if t.Length <= large then fprintf o "{%s}" (base64 t)
  else fprintf o "{%s...%4i}" (base64 (Array.sub t 0 (large-1))) t.Length

let length (x:bytes) = x.Length : int

let guid () =  "uuid:"^(System.Guid.NewGuid()).ToString()

let syms = ref 0


let sprint_raw (t:bytes) = 
  Array.fold_left (fun x y -> x ^ Printf.sprintf "%02x " y) "" t

let fprint_raw out t = 
  Array.iter (fun x -> Printf.fprintf out "%02x " x) t

let print_raw t = fprint_raw stdout t 

let nice_print_raw out t = 
  let n = 16 in
  let rec f (tt:bytes) = 
    let len = tt.Length in
    if len = 0 then
      ()
    else
    let linelen = min n len in	
    let ttt = Bytearray.sub tt 0 linelen in
    Printf.fprintf out "\t";
    fprint_raw out ttt;
    Printf.fprintf out "\n";
    f (Bytearray.sub tt linelen (len-linelen))
  in
  f t
	


let fprint_bytes out (str:string) (msg:bytes) =  
  Printf.fprintf out "%s (%d)\n" str msg.Length; 
  nice_print_raw out msg
  //Printf.fprintf out "\n"
  

let print_bytes (str:string) (msg:bytes) =  
  fprint_bytes stdout str msg


(* conversions between bytes and "basic" types: int, etc *)

let int_to_bytes nb i = 
  let rec put_bytes bb lb n = 
    if lb = 0 then failwith "not enough bytes"
    else 
      begin
        Bytearray.set bb (lb-1) (Byte.of_int (n%256));
	if n/256 > 0 then
	  put_bytes bb (lb-1) (n/256)
	else bb
      end
  in
  let b = Bytearray.make nb in
    put_bytes b nb i

let bytes_of_int nb i = int_to_bytes nb i

let int_of_bytes nb (t:bytes) = 
  if nb > t.Length then
    failwith "wrong parameter";
  let tt = Bytearray.sub t 0 nb in 
  List.fold_left (fun x y -> 256 * x + y) 0 (List.map Byte.to_int (Bytearray.to_list tt))



let intpair_of_bytes (msg: bytes) : (int*int) = 
  if msg.Length < 2 then
     failwith "wrong parameter";
  (Byte.to_int (Bytearray.get msg 0), Byte.to_int (Bytearray.get msg 1))

let bytes_of_intpair ((i,j): (int*int)) : bytes = 
  let msg = Bytearray.make 2 in
  Bytearray.set msg 0 (Byte.of_int i);
  Bytearray.set msg 1 (Byte.of_int j);
  msg
  
