(* Copyright (c) Microsoft Corporation.  All rights reserved.  *)

#light "off"

module Crypto

open System.Security.Cryptography
open Printf
open Data

type nonce = bytes
type seed = bytes
type guid = bytes

type key = 
    SymKey of bytes
  | RSAKey of RSACryptoServiceProvider

type mkey = key

type keyseed = bytes

let keytobytes kb = match kb with
    SymKey(k) -> k
  | _ -> failwith "keytobytes: asym, gss not implemented concretely"
 
type macs = bytes
type blocks = bytes

let symkey k = SymKey k

let keyseedtobytes : bytes -> bytes = fun n -> n
let bytestokeyseed : bytes -> bytes = fun n -> n
let noncetobytes : nonce -> bytes = fun n -> n
let bytestononce : bytes -> nonce = fun n -> n
let mactobytes : macs -> bytes = fun n -> n
let bytestomac : bytes -> macs = fun n -> n
let blockstobytes : blocks -> bytes = fun n -> n
let bytestoblocks : bytes -> blocks = fun n -> n


(* random number generation *) 

let rng = new RNGCryptoServiceProvider ()
let random () : str = 
  let x = Bytearray.make 16 in
  rng.GetBytes x; base64 x


(* let mkNonce224 () : bytes =
  let x = Bytearray.make 28 in
  rng.GetBytes x; x *)

let mkNonce () : bytes =
  let x = Bytearray.make 16 in
  rng.GetBytes x; x

let mkNonce256 () : bytes =
  let x = Bytearray.make 32 in
  rng.GetBytes x; x

let rand_seed () = mkNonce ()
let rand_nonce () = mkNonce256 ()
let rand_guid () = mkNonce ()


let mkRandom (nb:int) : bytes =
  let x = Bytearray.make nb in
  rng.GetBytes x; x



let mkKey () : bytes = (mkNonce())
let mkKey256 () : bytes = (mkNonce256())

let mkPassword () :str = 
  let x = Bytearray.make 7 in
  let b = rng.GetBytes x in
    base64 x

(*---- Hashing. *)

let md5_instance = MD5.Create ()
let md5 (x:bytes) : bytes = md5_instance.ComputeHash x

let sha1_instance = SHA1.Create ()
let sha1 (x:bytes) : bytes = sha1_instance.ComputeHash x

let sha256_instance = SHA256.Create ()
let sha256 (x:bytes) : bytes = sha256_instance.ComputeHash x

let hmacsha1 (k:key) (x:bytes) : bytes = 
  match k with 
      SymKey key -> (new HMACSHA1 (key)).ComputeHash x
    | _ -> failwith "HMAC-SHA1 only defined for symmetric keys"

let hmacmd5 (k:key) (x:bytes) : bytes = 
  match k with 
      SymKey key -> (new HMACMD5 (key)).ComputeHash x
    | _ -> failwith "HMAC-SHA1 only defined for symmetric keys"

//let hmacsha1Verify k v m =  
//  (hmacsha1 k v) = m


(*   WSE requires 16 bytes: http://www.dynamic-cast.com/mt-archives/000019.html
     one iteration: http://www.zvon.org/tmRFC/RFC2246/Output/chapter5.html *)
let psha1 (key:bytes) (seed:bytes) : key = 
  let x0 = concat (utf8 "WS-Security") seed in
    SymKey (Array.sub (hmacsha1 (SymKey key) (concat (hmacsha1 (SymKey key) x0) x0)) 0 16)


(*---- AES. *)



(* AES is block-based, so does implicit padding. To make decryption an inverse of
   encryption, we use concatvar, so as to prepend the length of the message. This
   is a waste if we know what the length is, but concatenating empty lists is
   symbolically tricky. We may provide a "raw AES" primitive if needed. *)
   
   
let pad plain = concatvar plain [| |]
let unpad plain = let v, pad = iconcat plain in v 

let aes_encrypt (k:key) (v:bytes) = 
  match k with 
      SymKey key ->
	let plain = v in
	let keyl = 8 * key.Length  in
	let block = 16 in (* 128 bits *)
	let rij = new RijndaelManaged() in
	  rij.set_KeySize(keyl);
	  //  rij.set_Padding(PaddingMode.ISO10126);
	  rij.GenerateIV();
	  let iv = rij.get_IV() in
	  let enc = rij.CreateEncryptor(key,iv) in
	  let mems = new System.IO.MemoryStream() in
	  let crs = new CryptoStream(mems,enc,CryptoStreamMode.Write) in
	  let _ =  crs.Write(plain,0,plain.Length) in
	  let _ = crs.FlushFinalBlock() in
	  let cip = mems.ToArray() in
	  let ctxt = Bytearray.make (iv.Length + cip.Length) in
	  let _ = System.Array.Copy(iv,0,ctxt,0,iv.Length) in
	  let _ = System.Array.Copy(cip,0,ctxt,iv.Length,cip.Length) in
	    mems.Close();
	    crs.Close();
	    ctxt
    | _ -> failwith "AES only defined for symmetric keys"

let aes_decrypt (k:key) (cipher:bytes) = 
  match k with 
      SymKey key ->
	let rij = new RijndaelManaged() in
	let keyl = 8 * key.Length  in
	  rij.set_KeySize(keyl);
	  let block = 16 in (* 128 bits *)
	  let iv = Bytearray.make (block) in
	  let _ = System.Array.Copy(cipher,0,iv,0,iv.Length) in
	  let cip = Bytearray.make (cipher.Length - iv.Length) in
	  let _ = System.Array.Copy(cipher,iv.Length,cip,0,(cipher.Length - iv.Length)) in
	  let dec = rij.CreateDecryptor(key,iv) in
	  let mems = new System.IO.MemoryStream(cip) in
	  let crs = new CryptoStream(mems,dec,CryptoStreamMode.Read) in
	  let plain = Bytearray.make(cipher.Length) in  
	  let _ =  crs.Read(plain,0,plain.Length) in
	    plain
    | _ -> failwith "AES only defined for symmetric keys"


let aes_encrypt_wiv (k:key) (iv:bytes) (plain:bytes) = 
  match k with 
      SymKey key ->
	let rij = new RijndaelManaged() in
	let keyl = 8 * key.Length  in
	rij.set_KeySize(keyl);
	(* default padding mode is PKCS7 *)
	let enc = rij.CreateEncryptor(key,iv) in
	let mems = new System.IO.MemoryStream() in
	let crs = new CryptoStream(mems,enc,CryptoStreamMode.Write) in
	let _ =  crs.Write(plain,0,plain.Length) in
	let _ = crs.FlushFinalBlock() in
	let cipher = mems.ToArray() in
	mems.Close();
	crs.Close();
	cipher
    | _ -> failwith "AES only defined for symmetric keys"

let aes_decrypt_wiv (k:key) (iv:bytes) (cipher:bytes) = 
  match k with 
      SymKey key ->
	let rij = new RijndaelManaged() in
	let keyl = 8 * key.Length  in
	rij.set_KeySize(keyl);
	rij.set_Padding(PaddingMode.None);
	//print_any (rij.get_Padding(),rij.get_BlockSize());
	let dec = rij.CreateDecryptor(key,iv) in
	let mems = new System.IO.MemoryStream(cipher) in
	let crs = new CryptoStream(mems,dec,CryptoStreamMode.Read) in
	let plain = Bytearray.make(cipher.Length) in  
	let _ =  crs.Read(plain,0,plain.Length) in
	plain
    | _ -> failwith "AES only defined for symmetric keys"


(* DES *)

let des_encrypt (k:key) (iv:bytes) (plain:bytes) = 
  match k with 
      SymKey key ->	
	let des = new DESCryptoServiceProvider() in
	//let keyl = 8 * key.Length  in
	//let block = 8 in (* 64 bits *)
	//des.set_KeySize(keyl);
	//print_any (des.get_Padding());
	(* default padding mode is PKCS7 *)
	let enc = des.CreateEncryptor(key,iv) in
	let mems = new System.IO.MemoryStream() in
	let crs = new CryptoStream(mems,enc,CryptoStreamMode.Write) in
	let _ =  crs.Write(plain,0,plain.Length) in
	let _ = crs.FlushFinalBlock() in 
	let cip = mems.ToArray() in
        mems.Close();
	crs.Close();
	cip
    | _ -> failwith "DES only defined for symmetric keys"


let des_decrypt (k:key) (iv:bytes) (cipher:bytes) = 
  match k with 
      SymKey key ->
	let des = new DESCryptoServiceProvider() in
	//let keyl = 8 * key.Length  in
	//des.set_KeySize(keyl);*)
	//let block = 8 in (* 64 bits *)
	//print_any (des.get_Padding());
	des.set_Padding(PaddingMode.None);
	let dec = des.CreateDecryptor(key,iv) in
	let mems = new System.IO.MemoryStream(cipher) in
	let crs = new CryptoStream(mems,dec,CryptoStreamMode.Read) in
	let plain = Bytearray.make(cipher.Length) in 
	let _ =  crs.Read(plain,0,plain.Length) in
	mems.Close();
	crs.Close();
	plain
    | _ -> failwith "DES only defined for symmetric keys"


(*---- X509. *)

let x509signcert (i:str) (s:str) (key:bytes) :bytes = failwith "cannot generate X.509 certs"

let x509pubkey (x:bytes) : bytes = 
  let c = new System.Security.Cryptography.X509Certificates.X509Certificate (x) in
  c.GetPublicKey()

let x509issuer (x:bytes) : str = 
  let c = new System.Security.Cryptography.X509Certificates.X509Certificate (x) in
  c.get_Issuer()

let x509subject (x:bytes) : str = 
  let c = new System.Security.Cryptography.X509Certificates.X509Certificate (x) in
  c.get_Subject()

let x509number (x:bytes) : str = 
  let c = new System.Security.Cryptography.X509Certificates.X509Certificate (x) in
  c.GetSerialNumberString()

let x509alg (x:bytes) : str = 
  let c = new System.Security.Cryptography.X509Certificates.X509Certificate (x) in
  c.GetKeyAlgorithm()


open System.Security.Cryptography.X509Certificates

let x509info (cert:bytes) = 
  printf "\n\nCertificate parameters\n\n";
  let c = new X509Certificate(cert) in
    //printf "\nToString:\n%s\n" (c.ToString());
    //print_bytes "\nSerialNumber" (c.GetSerialNumber());
    printf "Serial Number: %s\n" (c.GetSerialNumberString());
    //printf "GetName:\n%s\n" (c.GetName()); 	
    printf "Subject: %s\n" (c.get_Subject()); 	
    //printf "GetIssuerName:\n%s\n" (c.GetIssuerName()); 	
    printf "Issuer: %s\n" (c.get_Issuer()); 	
    printf "Effective Date: %s\n" (c.GetEffectiveDateString()); 	
    printf "Expiration Date: %s\n" (c.GetExpirationDateString());
    //print_bytes "\nGetRawCertData" (c.GetRawCertData());
    //print_bytes "CertHash" (c.GetCertHash());  	    
    printf "Cert Hash: %s\n" (c.GetCertHashString()); 	
    printf "HashCode: %x\n" (c.GetHashCode()); 	
    printf "Format: %s\n" (c.GetFormat()); 	
    printf "Key Algorithm: %s\n" (c.GetKeyAlgorithm()); 	
    //print_bytes "GetKeyAlgorithmParameters" (c.GetKeyAlgorithmParameters());
    printf "Key Algorithm Parameters: %s" (c.GetKeyAlgorithmParametersString());
    print_bytes "Public Key" (c.GetPublicKey()); 	
    //printf "\nGetPublicKeyString:\n%s\n" (c.GetPublicKeyString()); 
    ()


let getCN (s:string) = 
  let ss = String.split [',';' '] s in
  let rec f sl = 
    match sl with 
    | [] -> "" 
    | (h:string)::t ->   
       let ss = String.split ['='] h in
       if ss.Length = 2 && (List.hd ss = "CN") then
         (List.hd (List.tl ss))
       else
         f t
  in
    f ss



let debug = ref true
let set_debug d = debug := d

let print_msg str = 
  if !debug then print_string str

let x509verify (cert:bytes) s = 
  let c = new X509Certificate2(cert) in
  let ch = new X509Chain() in

  (* does it speed up verification?? *)
  ch.ChainPolicy.set_VerificationFlags(X509VerificationFlags.NoFlag);
  ch.ChainPolicy.set_RevocationMode(X509RevocationMode.NoCheck);
  let b = ch.Build(c) in	
  if !debug then	
    begin printf "Is the certificate `valid'? "; print_any b; printf "\n" end;

  let bad = ref false in
  let l = ch.ChainElements.get_Count() in
  print_msg (sprintf "Number of certificates in the chain: %d\n" l);
  for i = 0 to l-1 do
    let el = ch.ChainElements.Item(i) in
    for j = 0 to el.ChainElementStatus.get_Length() - 1 do
      bad := !bad || (el.ChainElementStatus.(j).get_Status() = X509ChainStatusFlags.NotSignatureValid);
    done
  done;
  let s2 = getCN (c.get_Subject()) in
  if s <> s2 then
    print_msg (sprintf "Warning: the server's url (%s) and the subject of the certificate (%s) differ.\n\n" s s2)
  else
    print_msg "The server's name (from the URL) and the Common Name of the subject of its certificate match.\n\n";
  (not !bad)


let x509analyse (cert:bytes) = 
    x509info cert;

    let c2 = new X509Certificate2(cert) in
    X509Certificate2UI.DisplayCertificate(c2);

    printf "\n\nCertificate verification\n\n";

    printf "X509Certificate2.Verify() gives\t";
    print_any (c2.Verify());
    printf "\n";

    let ch = new X509Chain() in
    ch.ChainPolicy.set_VerificationFlags(X509VerificationFlags.NoFlag);
    ch.ChainPolicy.set_RevocationMode(X509RevocationMode.NoCheck);

    printf "X509Chain.Build() gives\t";
    print_any (ch.Build(c2));
    printf "\n";

    printf "ChainPolicy.VerificationFlags\t";
    print_any (ch.ChainPolicy.get_VerificationFlags());
    printf "\n";

    printf "ChainPolicy.VerificationTime\t";
    print_any (ch.ChainPolicy.get_VerificationTime());
    printf "\n";

    printf "ChainPolicy.RevocationMode\t";
    print_any (ch.ChainPolicy.get_RevocationMode());
    printf "\n";

    printf "ChainPolicy.RevocationFlag\t";
    print_any (ch.ChainPolicy.get_RevocationFlag());
    printf "\n";

    let l = ch.ChainElements.get_Count() in
    printf "There are %d elements in the chain\n" l;

    let bad = ref false in
    for i = 0 to l-1 do
      //if (ch.ChainStatus.Length > 1) then
	let el = ch.ChainElements.Item(i) in
	printf "status for element %d : " (i+1);
	let sl = el.ChainElementStatus.get_Length() in
	if sl = 0 then
	  printf "ok\n"
	else
	  begin
	    printf "\n";
	    for j = 0 to sl - 1 do
	      let es = el.ChainElementStatus.(j) in
	      let st = es.get_Status() in
	      bad := !bad || (st = X509ChainStatusFlags.NotSignatureValid);
	      printf "\t";
              print_any (st);
	      printf "\t";
	      print_any (es.get_StatusInformation());
	      printf "\n"
            done
	  end
    done;
    printf "\n-------------------------------\n";
    (not !bad)


let x509choose () =
  let store = new X509Store(StoreName.My, StoreLocation.LocalMachine) in
  store.Open(OpenFlags.ReadOnly ||| OpenFlags.OpenExistingOnly);
  let collection = store.Certificates in
  let scollection = X509Certificate2UI.SelectFromCollection(collection, "Certificate Select", "Select a list of acceptable CAs", X509SelectionFlag.MultiSelection) in
  let dnamelist = ref (Bytearray.make 0) in
  for i = 0 to scollection.get_Count() - 1 do
    let x509 = scollection.get_Item(i) in
    let dname = new X500DistinguishedName(x509.IssuerName) in
    dnamelist := concat !dnamelist (dname.RawData);
    print_bytes "dnamelist" !dnamelist;
    printf "\nX500DistinguishedName: %s\n" dname.Name;
    x509.Reset();
  done;
  store.Close();
  !dnamelist



(*---- RSA. *)

let rsa_skey (key:str) = 
  let rsa = new RSACryptoServiceProvider () in
    rsa.FromXmlString(key);
    RSAKey rsa

let rsa_pub (k:key) = 
  match k with
      RSAKey skey ->
	let pub = skey.ExportParameters(false) in
	let rsa = new RSACryptoServiceProvider () in
	  rsa.ImportParameters(pub);
	  RSAKey rsa
    | _ -> failwith "Rsa_pub only defined for asymmetric private keys"

let rsa_pkey (key:str) = rsa_pub (rsa_skey key)

(* only works for 1024 bits keys *)
let rsa_pkey_bytes (key:byte[]) = 
  let mutable rkey = new RSAParameters() in
  rkey.Exponent <- Array.sub key (key.Length - 3) 3;
  rkey.Modulus <- Array.sub key (key.Length - 128 - 5) 128; 
  let rsa = new RSACryptoServiceProvider () in
    rsa.ImportParameters(rkey);
    RSAKey rsa

(* probably only works for 1024 bits keys *)
let rsa_skey_bytes (key:byte[]) = 
  let mutable rkey = new RSAParameters() in
  rkey.Modulus <- Array.sub key 11 128;
  rkey.Exponent <- Array.sub key 141 3;
  rkey.D <- Array.sub key 147 128;
  rkey.P <- Array.sub key 278 64;
  rkey.Q <- Array.sub key 345 64;
  rkey.DP <- Array.sub key 412 64;
  rkey.DQ <- Array.sub key 479 64;
  rkey.InverseQ <- Array.sub key 545 64; 
  let rsa = new RSACryptoServiceProvider () in
    rsa.ImportParameters(rkey);
    RSAKey rsa


let asympubkey k = (rsa_pkey_bytes k)
let asymprivkey k = (rsa_skey_bytes k)

let asympub k = (rsa_pkey k)
let asympriv k = (rsa_skey k)

let rsa_keygen () = 
  let rsa = new RSACryptoServiceProvider() in
    RSAKey rsa

(* RSA Encryption *)

(* Setting OAEP = true for indigo, = false for WSE *)
let rsa_encrypt (k:key)  (v:bytes)  = 
  match k with 
      RSAKey pkey ->  pkey.Encrypt(v,false) 
    | _ -> failwith "RSA encryption only defined for asymmetric keys"
    
let rsa_encrypt_oaep (k:key) (v:bytes)  = 
  match k with 
      RSAKey pkey ->  pkey.Encrypt(v,true) 
    | _ -> failwith "RSA encryption only defined for asymmetric keys"

let rsa_decrypt (k:key) (v:bytes) =
  match k with 
      RSAKey skey ->  skey.Decrypt(v,false) 
    | _ -> failwith "RSA encryption only defined for asymmetric keys"


let rsa_decrypt_oaep (k:key) (v:bytes) =
  match k with 
      RSAKey skey ->  skey.Decrypt(v,true) 
    | _ -> failwith "RSA encryption only defined for asymmetric keys"

(* RSA Signatures *)

let rsa_sign (k:key) (v:bytes)  = 
  match k with 
      RSAKey skey ->    
	let oid = CryptoConfig.MapNameToOID "sha1" in
	  skey.SignHash (sha1 v, oid)
    | _ -> failwith "RSA signature only defined for asymmetric keys"
    

let rsa_verify (k:key) (v:bytes) (x:bytes)  =
  match k with 
      RSAKey pkey ->    
	let oid = CryptoConfig.MapNameToOID "sha1" in
	  if not (pkey.VerifyHash (sha1 v, oid, x))
	  then failwith "RSA_verify failed"
    | _ -> failwith "RSA signature only defined for asymmetric keys"



(* ARCfour (see [draft-kaukonen-cipher-arcfour-03.txt]) *)

type arcfour_state = {i : int ref; j : int ref; s : int array}
//type stream_cipher_state = arcfour_state



let arcfour_init (key:byte[]) = 
  let s = Array.init 256 (fun i -> i) in
  let keylen = key.Length in
  let j = ref 0 in
  for i = 0 to 255 do
    j := (!j + s.(i) + (Byte.to_int (Bytearray.get key (i%keylen)))) &&& 0xff;
    let temp = s.(i) in
    s.(i) <- s.(!j);
    s.(!j) <- temp
  done;
  {i=ref 0;j=ref 0;s=s}

let print_array str (msg:int[]) =  
  Printf.printf "\n%s (%d)\n" str msg.Length; 
  Array.iter (fun x -> Printf.printf "%02x " x) msg


let arcfour_encrypt (p:byte[]) s = 
  let len = p.Length in
  let c = Bytearray.make len in      
  let i = s.i in
  let j = s.j in
  let ss = s.s in
  for l = 0 to len-1 do
    i := (!i+1) &&& 0xff;
    j := (!j + ss.(!i)) &&& 0xff;
    let temp = ss.(!i) in
    ss.(!i) <- ss.(!j);
    ss.(!j) <- temp;
    Bytearray.set c l (Byte.of_int 
		       ((ss.((ss.(!i) + ss.(!j)) &&& 0xff))
			 ^^^
		       (Byte.to_int (Bytearray.get p l))))
  done;
  //print_array "[arcfour_encrypt] ss" ss;
  c

type stream_cipher_state = ArcFourState of arcfour_state

let stream_cipher_init key = 
  let st = arcfour_init key in
  ArcFourState st

let stream_cipher_encrypt p st = 
  match st with
  | ArcFourState afst -> arcfour_encrypt p afst

(* used by TLS *)


let xor s1 s2 nb =
  if Bytearray.length s1 < nb or Bytearray.length s2 < nb then
    failwith "xor: arrays too short"
  else
    let res = Bytearray.make nb in  
    for i=0 to nb-1 do
      res.(i) <- Byte.of_int ((Byte.to_int (Bytearray.get s1 i)) ^^^ (Byte.to_int (Bytearray.get s2 i)))
    done;
    res

let p_sha1 secret seed nb = 
  let it = (nb/20)+1 in
  let res = Bytearray.make nb in
    let a = ref seed in
    for i=0 to it-2 do
      a := hmacsha1 secret !a;
      Bytearray.blit (hmacsha1 secret (concat !a seed)) 0 res (i*20) 20;
    done;
    a := hmacsha1 secret !a;
    let r=nb%20 in
    Bytearray.blit (hmacsha1 secret (concat !a seed)) 0 res (nb-r) r;
    res

let p_md5 secret seed nb = 
  let it = (nb/16)+1 in
  let res = Bytearray.make nb in
    let a = ref seed in
    for i=0 to it-2 do
      a := hmacmd5 secret !a; 
      Bytearray.blit (hmacmd5 secret (concat !a seed)) 0 res (i*16) 16
    done;
    a := hmacmd5 secret !a;
    let r=nb%16 in
    Bytearray.blit (hmacmd5 secret (concat !a seed)) 0 res (nb-r) r;
    res

(* nb is the number of bytes to be generated *)
let prf (secret:bytes) (label:bytes) (seed:bytes) (nb:int) =
  let l_s = Bytearray.length secret in
   let l_s1 = (l_s+1)/2 in
    let secret1 = Bytearray.sub secret 0 l_s1 and
        secret2 = Bytearray.sub secret (l_s-l_s1) l_s1 in
    let newseed = concat label seed in
        let hmd5 = p_md5 (symkey secret1) newseed nb and
            hsha1 = p_sha1 (symkey secret2) newseed nb
        in 
    xor hmd5 hsha1 nb

let prf_hmac (secret:bytes) (label:bytes) (seed:bytes) (nb:int) = 
  prf secret label seed nb

let checkprf_hmac (secret:bytes) (label:bytes) (seed:bytes) (nb:int) p = 
  p = prf secret label seed nb

let ssl_prf1 secret label seed = md5 (concat secret (sha1 (concat (utf8 label) (concat secret seed))))

let ssl_prf secret seed nb = 
  let gen_label i = str (String.make (i+1) (char_of_int ((int_of_char 'A') + i))) in
  let rec apply_prf res n = 
    if n > nb then 
      Bytearray.sub res 0 nb
    else apply_prf (concat res (ssl_prf1 secret (gen_label (n/16)) seed)) (n+16)
  in
  apply_prf (Bytearray.make 0) 0





(* for CV only: only the interfaces matters *)

let secret_nonce (n:nonce) = ()
let secret_bitstring (x:bytes) = ()

let msymkey (k:bytes) = SymKey k
let macsToBlocksize (m:bytes) = m
let blocksizeToMacs (m:bytes) = m

let bitstringToMacs (m:bytes) = m
let bitstringToBlocksize (m: bytes) = m
let blocksizeToBitstring (m:bytes) = m
let prf_cv (m:bytes) = m
let mprf_cv (m:bytes) = m

let kgen (m:bytes) = m
let mkgen (m:bytes) = m
let pkgen (seed:bytes) = SymKey seed

let dec (k:key) (c:bytes) = c
let enc (k:key) (m:bytes) (iv:bytes) = m

let deca (k:key) (c:bytes) = c
let enca (k:key) (m:bytes) (s:bytes) = m


let hmacsha1_verify (k:key) (msg:bytes) (m:macs) = 
	(hmacsha1 k msg) = m

let hmacmd5_verify (k:key) (msg:bytes) (m:macs) = 
	(hmacmd5 k msg) = m

let check mmsg mkey realtag =
  hmacsha1 (msymkey mkey) mmsg = realtag 


