open Data
open Formats_ciphersuites
open Formats


let zeros nb = Bytearray.make nb
 
let zeros_wlen nb =   
  let z = Bytearray.make (1+nb) in
  Bytearray.set z 0 (Byte.of_int nb);
  z

let seq_to_bytes c = bytes_of_int 8 c
let bytes_of_seq s = seq_to_bytes s

let int_of_char c =
  if ((Char.code '0')<= (Char.code c) && (Char.code c)<=(Char.code '9')) 
    then (Char.code c)-(Char.code '0')
  else
    if ((Char.code 'a')<= (Char.code c) && (Char.code c)<=(Char.code 'f')) 
      then 10+(Char.code c) - (Char.code 'a')
    else
      if ((Char.code 'A')<= (Char.code c) && (Char.code c)<=(Char.code 'F')) 
        then 10+(Char.code c) - (Char.code 'A')
      else 
        failwith "[int_of_char]"

let rec intlist_of_string str =
  let hex_of_pair (a,b) = ((int_of_char a)*16)+int_of_char b in
    let get_pair str = (String.get str 0),(String.get str 1) in 
      if String.length str > 1 then
	(hex_of_pair (get_pair str))::intlist_of_string (String.sub str 3 ((String.length str)-3)) 
      else
	[]



(*** constants ***)

let empty_bstr = zeros 0

let empty_sid = zeros 1

let ccs_msg = bytes_of_int 1 1


(* SSLv3 constants *)
 
let ssl_pad1_md5 = let a = Bytearray.make 48 in Bytearray.fill a 0 48 (Byte.of_int 0x36); a
let ssl_pad2_md5 = let a = Bytearray.make 48 in Bytearray.fill a 0 48 (Byte.of_int 0x5c); a
let ssl_pad1_sha1 = let a = Bytearray.make 40 in Bytearray.fill a 0 40 (Byte.of_int 0x36); a
let ssl_pad2_sha1 = let a = Bytearray.make 40 in Bytearray.fill a 0 40 (Byte.of_int 0x5c); a

let ssl_sender_client = Bytearray.of_list (List.map Byte.of_int [0x43; 0x4C; 0x4E; 0x54])
let ssl_sender_server = Bytearray.of_list (List.map Byte.of_int [0x53; 0x52; 0x56; 0x52])




(*** Versions ***)


let version_of_int i = 
  match i with
    | (0,2) -> SSLv2
    | (2,0) -> SSLv2
    | (3,0) -> SSLv3
    | (3,1) -> TLS1p0
    | (3,2) -> TLS1p1
    | (3,3) -> TLS1p2
    | _ -> failwith "[version_of_int] unknown version"

let int_of_version ver = 
  if ver = SSLv2 then (0,2)
  else if ver = SSLv3 then (3,0) 
  else if ver = TLS1p0 then (3,1)
  else if ver = TLS1p1 then (3,2)
  else if ver = TLS1p2 then (3,3)
  else failwith "unknown version"

let version_of_bytes (msg: bytes) =  msg
let bytes_of_version (ver:version) = ver

let ver2PV ver = 
  let (maj,min) = int_of_version ver in
  {major=maj;minor=min}

let PV2ver pv = version_of_int (pv.major,pv.minor)

let PV2pmver (pv:ProtocolVersion) = 
  let pmversion_of_int i = 
    match i with
    | (0,2) -> PM_SSLv2
    | (2,0) -> PM_SSLv2
    | (3,0) -> PM_SSLv3
    | (3,1) -> PM_TLS1p0
    | (3,2) -> PM_TLS1p1
    | (3,3) -> PM_TLS1p2
    | _ -> failwith "[pmversion_of_int] unknown version"
  in
  pmversion_of_int (pv.major,pv.minor)

(*** Content Types ***)

let ContentType_of_bytes (msg: bytes) =  msg
let bytes_of_ContentType (ct:ContentType) = ct



(*** Alerts ***)

let string_of_Alert a = 
  let (l,d) = (a.level,a.description) in
  let level = 
    match l with
    | AL_warning -> "warning"
    | AL_fatal -> "fatal"
    | AL_unknown_level x -> Printf.sprintf "unknown level (%d)" x
  in
  let desc = 
    match d with
    | AD_close_notify -> "close notify"
    | AD_unexpected_message -> "unexpected message"
    | AD_bad_record_mac -> "bad record mac"
    | AD_decryption_failed -> "decryption failed"
    | AD_record_overflow -> "record overflow"
    | AD_decompression_failure -> "decompression failure"
    | AD_handshake_failure -> "handshake failure"
    | AD_no_certificate -> "no certificate"
    | AD_bad_certificate -> "bad certificate"
    | AD_unsupported_certificate -> "unsupported certificate"
    | AD_certificate_revoked -> "certificate revoked"
    | AD_certificate_expired -> "certificate expired"
    | AD_certificate_unknown -> "certificate unknown"
    | AD_illegal_parameter -> "illegal parameter"
    | AD_unknown_ca -> "unknown ca"
    | AD_access_denied -> "access denied"
    | AD_decode_error -> "decode error"
    | AD_decrypt_error -> "decrypt error"
    | AD_export_restriction -> "export restriction"
    | AD_protocol_version -> "protocol version"
    | AD_insufficient_security -> "insufficient security"
    | AD_internal_error -> "internal error"
    | AD_user_cancelled -> "user cancelled"
    | AD_no_renegotiation -> "no renegotiation"
    | AD_unknown_description x ->  Printf.sprintf "unknown description (%d)" x
  in Printf.sprintf "%s (%s)" level desc

let intpair_of_AlertDescription ad = 
  match ad with
    | AD_close_notify ->            (1, 0)
    | AD_unexpected_message ->      (2,10)
    | AD_bad_record_mac ->          (2,20)
    | AD_decryption_failed ->       (2,21)
    | AD_record_overflow ->         (2,22)
    | AD_decompression_failure ->   (2,30)
    | AD_handshake_failure ->       (2,40)
    | AD_no_certificate ->          (1,41)
    | AD_bad_certificate ->         (1,42)
    | AD_unsupported_certificate -> (1,43)
    | AD_certificate_revoked ->     (1,44)
    | AD_certificate_expired ->     (1,45)
    | AD_certificate_unknown ->     (1,46)
    | AD_illegal_parameter ->       (2,47)
    | AD_unknown_ca ->              (2,48)
    | AD_access_denied ->           (2,49)
    | AD_decode_error ->            (2,50)
    | AD_decrypt_error ->           (1,51)
    | AD_export_restriction ->      (2,60)
    | AD_protocol_version ->        (2,70)
    | AD_insufficient_security ->   (2,71)
    | AD_internal_error ->          (2,80)
    | AD_user_cancelled ->          (1,90)
    | AD_no_renegotiation ->        (1,100)
    | AD_unknown_description x ->   (0,x)


let Alert_of_intpair (l,d) = 
  let level = 
   match l with 
    | 1 -> AL_warning
    | 2 -> AL_fatal
    | x -> (AL_unknown_level x)
  in 
  let desc = 
    match d with
    |  0 -> AD_close_notify 
    | 10 -> AD_unexpected_message 
    | 20 -> AD_bad_record_mac 
    | 21 -> AD_decryption_failed 
    | 22 -> AD_record_overflow 
    | 30 -> AD_decompression_failure 
    | 40 -> AD_handshake_failure 
    | 41 -> AD_no_certificate 
    | 42 -> AD_bad_certificate 
    | 43 -> AD_unsupported_certificate 
    | 44 -> AD_certificate_revoked 
    | 45 -> AD_certificate_expired 
    | 46 -> AD_certificate_unknown 
    | 47 -> AD_illegal_parameter 
    | 48 -> AD_unknown_ca 
    | 49 -> AD_access_denied 
    | 50 -> AD_decode_error 
    | 51 -> AD_decrypt_error 
    | 60 -> AD_export_restriction 
    | 70 -> AD_protocol_version 
    | 71 -> AD_insufficient_security 
    | 80 -> AD_internal_error 
    | 90 -> AD_user_cancelled 
    | 100 -> AD_no_renegotiation 
    | x -> (AD_unknown_description x)
  in {level = level; description = desc }

let Alert_of_bytes msg = Alert_of_intpair (intpair_of_bytes msg)

let bytes_of_Alert a = 
  bytes_of_intpair (intpair_of_AlertDescription a.description) 

let bytes_of_AlertDescription ad = 
  bytes_of_intpair (intpair_of_AlertDescription ad) 


(*** Handshake Types ***)

let bytes_of_HandshakeType (ht: HandshakeType) = ht
let HandshakeType_of_bytes (msg: bytes) = msg


(*** Ciphersuites and compression methods ***)

let bytes_of_CipherSuite cs : Data.bytes = 
  bytes_of_intpair (intpair_of_CipherSuite cs)

let CipherSuite_of_bytes (msg: Data.bytes) = 
  if (length msg) <> 2 then
    raise TLSException_decode_error;
  CipherSuite_of_intpair (intpair_of_bytes msg)


let bytes_of_CipherSuites csl : bytes = 
  let csl_len = 2 * List.length csl in
  let cslbytes = Bytearray.make (2+csl_len) in
  Bytearray.set cslbytes 0 (Byte.of_int (csl_len/256));
  Bytearray.set cslbytes 1 (Byte.of_int (csl_len%256));
  let put_cs msg start cs =
    let (a,b) = intpair_of_CipherSuite cs in
    Bytearray.set msg start (Byte.of_int a);
    Bytearray.set msg (start+1) (Byte.of_int b);
    msg
  in
  let rec put_ciphersuites msg l start = 
    match l with 
    | h::t -> put_ciphersuites (put_cs msg start h) t (start+2)
    | [] -> msg
  in
  put_ciphersuites cslbytes csl 2
    
let CipherSuites_of_bytes (msg: bytes) = 
  let msg_len = length msg in
  let csl_len = int_of_bytes 2 msg in
  if ((csl_len + 2) <> msg_len) || (msg_len % 2 = 1) then
    raise TLSException_decode_error;
  let rec get_ciphersuites i = 
    if i < msg_len then 
      (intpair_of_bytes (Bytearray.sub msg i 2))::get_ciphersuites (i+2)
    else [] 
  in
  (List.map CipherSuite_of_intpair (get_ciphersuites 2))




let bytes_of_CompressionMethod (cm:CompressionMethod) : bytes = 
  match cm with 
    CM_null -> bytes_of_int 1 0

let CompressionMethod_of_bytes (msg: bytes) : CompressionMethod = 
  if (length msg) <> 1 then
    raise TLSException_decode_error;
  if (int_of_bytes 1 msg) = 0 then
    CM_null
  else
    failwith "unknown compression method"

    
let bytes_of_CompressionMethods (cml:CompressionMethods) : bytes = 
  match cml with
  | [CM_null] -> bytes_of_int 2 256 (* i.e. 01 00 *) 
  | _ -> raise TLSException_internal_error
	
let CompressionMethods_of_bytes (msg: bytes) : CompressionMethods = 
  let msg_len = length msg in
  if msg_len < 1 then
    raise TLSException_decode_error;
  let cml_len = int_of_bytes 1 msg in
  if (cml_len + 1) <> msg_len then
    raise TLSException_decode_error;
  let rec get_compmethods i = 
    if i < msg_len then 
      (int_of_bytes 1 (Bytearray.sub msg i 1))::get_compmethods (i+1)
    else [] 
  in
  (List.map (fun x -> if x=0 then CM_null else failwith "unknown compression method") (get_compmethods 1))


(*** parsing functions ***)

let print_list l = 
  printf "\n[";
  let rec print_list_rec ll =
    match ll with
    | [] -> printf ""  
    | (i,j)::t -> 
	begin
	  printf "(%d,%d), " i j;
	  print_list_rec t
	end
  in
  print_list_rec l;
  printf "]\n"

(* the message will be decomposed into len.Length+1 fields;
 * the length of the j-th field is either given by 
 * the first component of the j-th element in len,
 * or by the some i-th field 
 * where the i-th element in len is (-l,j);
 * element (-l,0) means that the first l bytes give
 * the length of the corresponding field
 * the last field is what is left after parsing (may be the empty bitstring)
 *)
let parse_bytes (msg: bytes) (lengths: (int*int) list) : (bytes list) = 
  let rec replace i lengths v = 
    if i < 1 then
      raise TLSException_internal_error
    else 
      match lengths with
      | h::t -> if i = 1 then v::t else replace (i-1) t v
      | [] -> raise TLSException_internal_error
  in    
  let rec parse_bytes_rec msg msg_len lengths j = 
    //print_bytes "\n\n[parse_bytes]" msg;
    //printf "\nlen=%d j=%d" msg_len j;
    //print_list lengths;
    match lengths with
    | (l,i)::t -> 
	if l >= 0 then
	  begin
	    if l > msg_len then 
	      raise TLSException_decode_error;
	    (Bytearray.sub msg 0 l)
	    :: (parse_bytes_rec (Bytearray.sub msg l (msg_len-l)) (msg_len-l) t (j+1))
	  end
	else
	  begin
	    let l = -l in
	    if l > msg_len then 
	      raise TLSException_decode_error;
	    let b_field_len = Bytearray.sub msg 0 l in
	    let field_len = int_of_bytes l b_field_len in
	    if l+field_len > msg_len then 
	      raise TLSException_decode_error;
	    if i = 0 then
	      let field = Bytearray.sub msg l field_len in
	      (concat b_field_len field)
	      :: (parse_bytes_rec (Bytearray.sub msg (l+field_len) (msg_len-(l+field_len))) (msg_len-(l+field_len)) t (j+1))
	    else
	      let len_bis = replace (i-j) t (field_len,0) in
	      b_field_len :: (parse_bytes_rec (Bytearray.sub msg l (msg_len-l)) (msg_len-l) len_bis (j+1))
	  end
    | [] -> [msg]
  in
  parse_bytes_rec msg (msg.Length) lengths 1


let parseClientHello (msg: bytes) = 
  match parse_bytes msg [(2,0);(32,0);(-1,0);(-2,0);(-1,0)] with
  | [b_ver; cr; sid; b_csl; b_cml; extra] -> (b_ver, cr, sid, b_csl, b_cml, extra)
  | _ -> raise TLSException_decode_error

let parseServerHello (msg: bytes) =
  match parse_bytes msg [(2,0);(32,0);(-1,0);(2,0);(1,0)] with
  | [b_ver; sr; sid; b_cs; b_cm; extra] -> 
      if extra = empty_bstr then
	(b_ver, sr, sid, b_cs, b_cm)
      else
	raise TLSException_decode_error
  | _ -> raise TLSException_decode_error

let parsePMS (pms: bytes) = 
  match parse_bytes pms [(2,0)] with
  | [prefix;random] -> prefix,random
  | _ -> raise TLSException_decode_error



let parseHandshake (msg: bytes) = 
  match parse_bytes msg [(1,0);(-3,3);(0,0)] with
    | [b_msg_type; b_length; b_body; extra] -> 
	if extra = empty_bstr then
	  (b_msg_type, b_length, b_body)
	else
	  raise TLSException_decode_error
    | _ -> raise TLSException_internal_error
  
let get_last_block bs msg = 
  Bytearray.sub msg (msg.Length - bs) bs

let parseRecord (msg:bytes) =
   match parse_bytes msg [(1,0);(2,0);(-2,4);(0,0)] with
    | [bct; bver; blen; msg; extra] -> 
	if extra = empty_bstr then
	  (bct, bver, blen, msg)
	else
	  raise TLSException_decode_error
    | _ -> raise TLSException_decode_error





(*** other functions ***)

let lt_ver v1 v2 = 
  let (maj1,min1) = (v1.major, v1.minor) in
  let (maj2,min2) = (v2.major, v2.minor) in
  (maj1 < maj2) || ((maj1 = maj2) && (min1 < min2))
  
let leq_ver v1 v2 = 
  (v1 = v2) || (lt_ver v1 v2)

let min_version (ver1:ProtocolVersion) (ver2:ProtocolVersion) = 
  if lt_ver ver1 ver2 then
    ver1
  else
    ver2

let selectVersion clver svermin svermax = 
  if lt_ver clver svermin then
    failwith "protocol_version"
  else
    min_version clver svermax 
      


let rec selectCipherSuite (csl:CipherSuites) (pref_csl:CipherSuites) = 
  match pref_csl with
  | [] -> raise TLSException_handshake_failure
  | h::t -> if List.mem h csl then bytes_of_CipherSuite h else selectCipherSuite csl t


let selectCompressionMethod cml = 
  if List.mem CM_null cml then
    CM_null
  else
    failwith "handshake failure"
      




