open Data
open Crypto
open Db
//open Cell

open Formats_ciphersuites
open Formats
open Conversions_lowlevel
open Conversions
open Record




type ServerName = str 
type CertName = str


(* corresponds to the Certified module in CCS08 paper *)
#if cv (* 1 *)
let md5 a = a
let sha1 a = a
let rsa_encrypt pk msg = 
  enca msg pk (rand_seed())
let rsa_decrypt sk msg =
  deca msg sk
let pkseed = Pi.name "keyseed"
let asympubkey a = pkgen(pkseed)
let random = rand_nonce()
let printHandshake hs = printf "\n" 
let print_msg str =   ()
#endif (* 1 *)
(* end *)




(*** sessions ***)

type SessionId = bytes * ConnectionEnd

type Session = {
    sid: bytes;
    ms: bytes;
    server_cert: Certificate;
    ch: ClientHello;
    sh: ServerHello;
    pms: bytes;
}

let cSessionsDb = create()

let getcSession session_id = 
  select cSessionsDb session_id

let storecSession session_id session = 
  insert cSessionsDb session_id session


let sSessionsDb = create()

let getsSession session_id = 
  select sSessionsDb session_id

let storesSession session_id session = 
  insert sSessionsDb session_id session


let mkSession sid cert pms ms ch sh =
  {
   sid = sid;
   server_cert = cert;
   pms = pms;
   ms = ms;
   ch = ch;
   sh = sh;
  }

(*** events ***)

type event = 
  | LeakedSecret of ConnectionEnd * Session * Connection

  | SendFinished of ConnectionEnd  * Session * Connection
  | AcceptFinished of ConnectionEnd * Session * Connection  

  | SendFinishedRes of ConnectionEnd * Session * Connection * ClientHello * ServerHello
  | AcceptFinishedRes of ConnectionEnd * Session * Connection * ClientHello * ServerHello
  
(*
(*--- hsEventsBegin *)
  SendFinished(entity, session, conn) 
  AcceptFinished(entity, session, conn) 
(*--- hsEventsEnd *)
(*--- hsResEventsBegin *)
  SendFinishedRes(entity, session, conn, ch, sh) 
  AcceptFinishedRes(entity, session, conn, ch, sh) 
(*--- hsResEventsEnd *)
*)


let tr = Pi.trace()

let chksecrecyChan = Pi.chan "chksecrecyChan"


let getKey cipher_state = 
  match cipher_state with
  | BlockCipherState (ek,iv) -> ek,iv 
  | _ -> failwith "stream ciphers not considered for now"


let checkSecrecy entity sess conn = 
  let pms = sess.pms in
  let ms = sess.ms in
  let read = conn.read in
  let write = conn.write in
  let rek,riv = getKey read.cipher_state in
  let rmk = read.mk in
  let wek,wiv = getKey write.cipher_state in
  let wmk = write.mk in
  let guess = Pi.recv chksecrecyChan in
  if guess = pms then 
    Pi.log tr (LeakedSecret(entity,sess,conn))
  else if guess = ms then
    Pi.log tr (LeakedSecret(entity,sess,conn))
  else if guess = rek then
    Pi.log tr (LeakedSecret(entity,sess,conn))
  else if guess = rmk then
    Pi.log tr (LeakedSecret(entity,sess,conn))
  else if guess = wek then
    Pi.log tr (LeakedSecret(entity,sess,conn))
  else if guess = wmk then
    Pi.log tr (LeakedSecret(entity,sess,conn))
  else 
    failwith  "checksecrecy error"


let callCheckSecrecy entity sess conn = 
#if fs
  ()
#else
#if hssecr
  Pi.fork (fun () -> checkSecrecy entity sess conn)
#else
  ()
#endif
#endif

(*** databases ***)

let createState () = create()
let getState x y = select x y 
let storeState x y z = insert x y z

let hmDb = createState()
let finDb = createState()
let finResDb = createState()
let pwdAppDb = createState()





(*** pretty-printing functions ***)

#if fs
let debug = ref true
let set_debug d = debug := d
#endif


let print_msg str = 
#if fs
  if !debug then print_string str
#else
  ()
#endif







(*** functions for "buffered receving" ***)

#if concrete      

let hsRecv conn (buf:bytes) (checkver:bool) = 
  let conn, buf = 
    if buf.Length = 0 then 
      recv_hs conn checkver
    else
      conn,buf
  in
  if buf.Length < 4 then 
    raise TLSException_decode_error;
  let hslen = int_of_bytes 3 (Bytearray.sub buf 1 3) in
  let rec hs_recv conn (buf:bytes) = 
    if buf.Length < 4 + hslen then 
      let conn,msg = recv_hs conn checkver in
	hs_recv conn (concat buf msg)
    else
      conn,buf
  in
  let conn,buf = hs_recv conn buf in
    conn, Bytearray.sub buf 0 (4+hslen), Bytearray.sub buf (4+hslen) (buf.Length - (4+hslen))

#else

let hsRecv conn (buf:bytes) checkver = 
  let conn,msg = recv_hs conn checkver in
    conn,msg,empty_bstr

#endif 


let hsRecv_null conn (buf:bytes) checkver =
#if concrete
  let conn,msg,buf = hsRecv conn buf checkver in
#else
  let msg = recv_null conn.net_conn CT_handshake in
#endif
    conn,msg,buf



let sendccs conn ccs_msg = 
#if concrete 
  let conn = send_ccs conn ccs_msg in
#else
  send_null conn.net_conn conn.crt_version CT_change_cipher_spec ccs_msg;
#endif
  conn

let recvccs conn = 
#if concrete 
  let conn,msg = recv_ccs conn in
#else
  let msg = recv_null conn.net_conn CT_change_cipher_spec in
#endif
  conn, msg


(*** helper functions ***)

let decompClientHello ch = 
    ch.client_version, 
    ch.ch_random, 
    ch.ch_session_id, 
    ch.cipher_suites, 
    ch.compression_methods, 
    ch.extra_data

let decompServerHello sh = 
    sh.server_version, 
    sh.sh_random, 
    sh.sh_session_id, 
    sh.cipher_suite, 
    sh.compression_method

let decompSession sess = 
  sess.sid,
  sess.ms,
  sess.server_cert,
  sess.ch,
  sess.sh,
  sess.pms



let bldHandshake (msg_type:HandshakeType) (body: HandshakeBody) : Handshake = 
  {msg_type = msg_type; length = 0; body = body }
  (* we will compute the length later when we'll deal with the bytes *)


let sendHandshake conn ht hsbody = 
  let hs = bytes_of_Handshake (bldHandshake ht hsbody) in
  let conn = send_hs conn hs in
  conn, hs

let sendHandshake_null conn ht hsbody = 
  let hs = bytes_of_Handshake (bldHandshake ht hsbody) in
#if concrete
  let conn = send_hs conn hs in
#else  
  send_null conn.net_conn conn.crt_version CT_handshake hs;
#endif
  conn, hs



let recvClientHello conn buf =
  let conn,msg,buf = hsRecv_null conn buf false in
  let (ht, len, body) = parseHandshake msg in
  if HT_client_hello = HandshakeType_of_bytes ht then
    let hs = ClientHello_of_bytes body in
    conn, msg, hs, buf
  else
    failwith "unexpected message"

let recvServerHello conn buf =
  let conn,msg,buf = hsRecv_null conn buf false in
  let (ht, len, body) = parseHandshake msg in
  if HT_server_hello = HandshakeType_of_bytes ht then
    let hs = ServerHello_of_bytes body in
    conn, msg, hs, buf
  else
    failwith "unexpected message"


let recvCertificate conn buf =
  let conn,msg,buf = hsRecv_null conn buf true in
  let (ht, len, body) = parseHandshake msg in
  if HT_certificate = HandshakeType_of_bytes ht then
    let hs = Certificate_of_bytes body in
    conn, msg, hs, buf
  else
    failwith "unexpected message"

let recvServerHelloDone conn buf =
  let conn,msg,buf = hsRecv_null conn buf true in
  let (ht, len, body) = parseHandshake msg in
  if HT_server_hello_done = HandshakeType_of_bytes ht then
    let hs = ServerHelloDone_of_bytes body in
    conn, msg, hs, buf
  else
    failwith "unexpected message"

let recvClientKeyExchange conn buf =
  let conn,msg,buf = hsRecv_null conn buf true in
  let (ht, len, body) = parseHandshake msg in
  if HT_client_key_exchange = HandshakeType_of_bytes ht then
    let hs = ClientKeyExchange_of_bytes body in
    conn, msg, hs, buf, len
  else
    failwith "unexpected message"

let recvHsFinished conn buf =
#if cv
  let conn,msg,buf = hsRecv_null conn buf true in
#else
  let conn,msg,buf = hsRecv conn buf true in
#endif
  let (ht, len, body) = parseHandshake msg in
  if HT_finished = HandshakeType_of_bytes ht then
    let hs = Finished_of_bytes body in
    conn, msg, hs, buf
  else
    failwith "unexpected message"



(* create ClientHello *)
let bldClientHello cver sid ccsl : ClientHello = 
  { 
    client_version = cver;
    ch_random = noncetobytes (rand_nonce()); 
    ch_session_id = sid;
    cipher_suites = ccsl;
    compression_methods = cml_null;
    extra_data = empty_bstr;
  }


(* helper functions for creating ServerHello *)




let bldSid () = 
  concat (bytes_of_int 1 32) (noncetobytes (rand_nonce()))




let new_session ver cm cs = 
  let sid =  bldSid () in 
  (None, sid)

let resume_session sid = 
  let sess = getsSession (sid,Server) in
  (Some sess),sid 


let try_resume_session sid ver cm cs = 
#if fs
  try 
    Some (getsSession (sid,Server)), sid
  with exn -> new_session ver cm cs
#else
  failwith "not used in attacker code"
#endif


let selectParams ch sver_min sver_max pref_csl = 
  let sver = selectVersion ch.client_version sver_min sver_max in
  let cs = selectCipherSuite ch.cipher_suites pref_csl in
  let cm = selectCompressionMethod ch.compression_methods in
  sver, cs, cm

let bldServerHello sver sid cs cm = 
  { server_version = sver;
    sh_random = noncetobytes(rand_nonce()); 
    sh_session_id = sid;
    cipher_suite = cs;
    compression_method = cm; }


let bldNewServerHello ch sver_min sver_max pref_csl =
  let sver, cs, cm = selectParams ch sver_min sver_max pref_csl in
  let session, sid = new_session sver cm cs in
  let sh = bldServerHello sver sid cs cm in
  sh, session

let bldResServerHello ch sver_min sver_max pref_csl =
  let sver, cs, cm = selectParams ch sver_min sver_max pref_csl in
  let session, sid = resume_session ch.ch_session_id in
  let sh = bldServerHello sver sid cs cm in
  sh, session

let bldTryResServerHello ch sver_min sver_max pref_csl resume = 
  let sver, cs, cm = selectParams ch sver_min sver_max pref_csl in
  let session, sid = 
    if resume then
      try_resume_session ch.ch_session_id sver cm cs
    else
      new_session sver cm cs
  in
  let sh = bldServerHello sver sid cs cm in
  sh, session


let bldEncryptedPreMasterSecret ver public_key pms = 
  let encpms = rsa_encrypt public_key pms in
  let epms = 
#if concrete 
    if (PV2ver ver) = SSLv3 then (* just for interoperability *)
      encpms
    else
#endif
      concat (bytes_of_int 2 (length encpms)) encpms
  in
    { pre_master_secret = epms }


(* supposes that the list contains only one certificate *)
let bldCerts (asinglecert: bytes) = 
#if concrete
  let len = length asinglecert in 
  let cert = Bytearray.make (6 + len) in
  Bytearray.blit (bytes_of_int 3 (len+3)) 0 cert 0 3;
  Bytearray.blit (bytes_of_int 3 (len)) 0 cert 3 3;
  Bytearray.blit asinglecert 0 cert 6 asinglecert.Length;
  { certificate_list = cert }
#else
  { certificate_list = asinglecert }
#endif

let getCerts (cert: Certificate) = 
#if concrete
  let chain = cert.certificate_list in
  let len =  int_of_bytes 3 (Bytearray.sub chain 3 3) in
  Bytearray.sub chain 6 len
#else
  cert.certificate_list
#endif

let compute_master_secret pms ver crandom srandom = 
#if concrete
  match PV2pmver ver with 
  | PM_SSLv3 -> ssl_prf pms (concat crandom srandom) 48
  | _ -> prf pms (utf8 (str "master secret")) (concat crandom srandom) 48
#else
  prf pms (utf8 (str "master secret")) (concat crandom srandom) 48
#endif


let compute_key_block ver ms crandom srandom nb = 
#if cv (* 6 *)
()
#else
#if concrete
  match PV2pmver ver with 
  | PM_SSLv3 -> ssl_prf ms (concat srandom crandom) nb
  | _ -> prf ms (utf8 "key expansion") (concat srandom crandom) nb
#else
  prf ms (utf8 (str "key expansion")) (concat srandom crandom) nb
#endif
#endif (* 6 *)
	

#if concrete

let split_key_block key_block hsize ksize ivsize = 
  let cmk = Bytearray.sub key_block 0 hsize in
  let smk = Bytearray.sub key_block hsize hsize in
  let cek = Bytearray.sub key_block (2*hsize) ksize in
  let sek = Bytearray.sub key_block (2*hsize+ksize) ksize in
  let civ = Bytearray.sub key_block (2*hsize+2*ksize) ivsize in
  let siv = Bytearray.sub key_block (2*hsize+2*ksize+ivsize) ivsize in
  (cmk,smk,cek,sek,civ,siv)

#else

let split_key_block key_block hsize ksize ivsize =   
  (
    Sub (key_block, 0),
    Sub (key_block, 1),
    Sub (key_block, 2),
    Sub (key_block, 3),
    Sub (key_block, 4),
    Sub (key_block, 5)
  )

#endif


#if concrete 

let get_keyblocks_sizes (sp:SecurityParameters) ver = 
  let hsize = 
    match sp.mac_algorithm with
    | MA_null -> 0
    | MA_sha1 -> 20
    | MA_md5 -> 16
  in
  let ksize = 
    match sp.bulk_cipher_algorithm with
    | BCA_null -> 0
    | BCA_des -> 8
    | BCA_aes -> 16
    | BCA_rc4 -> 16
    | _ -> failwith "not implemented"
  in
  let ivsize = 
    if PV2ver ver <> TLS1p1 then 
      ksize
    else
      0
  in
  hsize,ksize,ivsize

#else

let get_keyblocks_sizes (sp:SecurityParameters) (ver:ProtocolVersion) = (0,0,0)

#endif

let compute_nb hsize ksize ivsize = 
#if concrete
  2 * (hsize + ksize + ivsize)
#else
  0
#endif
  

(* it in fact computes the new connection states *)
let generateKeys entity cr sh ms = 
  let ver,sr,_,cs,_ = decompServerHello sh in
  let cs = CipherSuite_of_bytes cs in
  match cs with 
  | TLS_NULL_WITH_NULL_NULL -> failwith "internal error"
#if concrete
  | _ -> 
#else 
  | TLS_RSA_WITH_AES_128_CBC_SHA -> 
#endif
  let sp = init_sp cs in
  let version = PV2ver ver in
  (* RFC2246 uses the field IV_size without defining it *)
  let hsize,ksize,ivsize = get_keyblocks_sizes sp ver in
  let nb = compute_nb hsize ksize ivsize in
  let key_block = compute_key_block ver ms cr sr nb in
  let (cmk,smk,cek,sek,civ,siv) = split_key_block key_block hsize ksize ivsize in 
  let civ,siv = 
#if concrete
    if version = TLS1p1 then
      let iv = mkRandom ksize in
      iv,iv
    else
#endif
      civ,siv
  in
  let rmk,rek,riv,wmk,wek,wiv = 
    match entity with 
      | Client -> smk,sek,siv,cmk,cek,civ
      | Server -> cmk,cek,civ,smk,sek,siv
  in
  let rcipher_state, wcipher_state =       
    match sp.cipher_type with
      | CT_block -> BlockCipherState (rek,riv), BlockCipherState (wek,wiv) 
      | CT_stream -> StreamCipherState (stream_cipher_init rek), StreamCipherState (stream_cipher_init wek)
  in 
  let readst = { cipher_state = rcipher_state; mk = rmk; seq_num = 0; sparams = sp; } in 
  let writest = { cipher_state = wcipher_state; mk = wmk; seq_num = 0; sparams = sp; } in
  readst,writest

  
let bldVerifyData ver ms entity hsmsgs = 
#if concrete
  match PV2pmver ver with 
  | PM_SSLv3 ->
      let ssl_sender = 
	match entity with
	| Client -> ssl_sender_client 
	| Server -> ssl_sender_server
      in let mm = concat hsmsgs (concat ssl_sender ms) in
      concat 
	(md5 (concat ms (concat ssl_pad2_md5 (md5 (concat mm ssl_pad1_md5)))))
	(sha1 (concat ms (concat ssl_pad2_sha1 (sha1 (concat mm ssl_pad1_sha1)))))
  | _ -> 
#endif
      let tls_label = 
	match entity with
	| Client -> utf8 (str "client finished")
	| Server -> utf8 (str "server finished")
      in prf_hmac ms tls_label (concat (md5 hsmsgs) (sha1 hsmsgs)) 12




(*--- principalsBegin *)
let principals sess p1 p2 = 
  let subj = x509subject (getCerts sess.server_cert) in
  if p1 = subj then
    if p2 = sess.pms then
      ()
    else failwith "wrong client"
  else failwith "wrong server"
(*--- principalsEnd *)

(* functional correctness *)
let correctParam sess = 
  let sh = sess.sh in
  if sess.sid = sh.sh_session_id then
    ()
  else failwith "internal error (sid)"

let connInSess conn sess = 
  if conn.sessid = sess.sid then
    ()
  else
    failwith "session ids do not match"


let matchingResParam sess ch sh = 
  let sid = sess.sid in
  let init_sh = sess.sh in
  if sid = ch.ch_session_id then
    if sid = sh.sh_session_id then
      if sid = init_sh.sh_session_id then
	if sh.cipher_suite = init_sh.cipher_suite then
	  ()
	else failwith "internal error (ciphersuite downgrade)"
      else failwith "internal error (initial sh_session_id)"
    else failwith "internal error (sh_session_id)"
  else failwith "internal error (ch_session_id)"


let matchingResParamVer sess ch sh = 
  let sid = sess.sid in
  let init_sh = sess.sh in
  if sid = ch.ch_session_id then
    if sid = sh.sh_session_id then
      if sid = init_sh.sh_session_id then
	if sh.cipher_suite = init_sh.cipher_suite then
	  if sh.server_version = init_sh.server_version then () else failwith "version downgrade!"
	else failwith "internal error (ciphersuite downgrade)"
      else failwith "internal error (initial sh_session_id)"
    else failwith "internal error (sh_session_id)"
  else failwith "internal error (ch_session_id)"




(*** handshake phases ***)


let clientHelloMsgs conn ver sid ccsl : (Connection * ClientHello * ServerHello * bytes * bytes) = 
  (* send ClientHello *)
  let ch = bldClientHello ver sid ccsl in

  printClientHello "" ch;
  print_msg "sending ClientHello\n";
  let conn,hsClientHello = sendHandshake_null conn HT_client_hello (HB_client_hello ch)  in

  (* receive ServerHello *)
  let conn, hsServerHello, sh, buf = recvServerHello conn empty_bstr in
  printHandshake hsServerHello;
  printServerHello "" sh;

  let conn = updateConn_version conn sh.sh_session_id sh.server_version in
  let hsm = concat hsClientHello hsServerHello in

  (conn,ch,sh,buf,hsm)

let clientHelloMsgs_att net_conn ver sid ccsl = 
  let conn = initConn net_conn Client (ver2PV TLS1p0) in
  let (conn,ch,sh,buf,hsm) = clientHelloMsgs conn ver sid ccsl in
  let cr = ch.ch_random in
  let cell = (conn,ch,sh,buf,hsm) in
  storeState hmDb ("cHMphase",cr) cell;
  cr


let gen_csl cs = 
#if concrete
  [CipherSuite_of_bytes cs]
#else
  (cs,cs)
#endif

let clientHelloMsgsRes_att net_conn sessid init_ver = 
  let id,entity = sessid in
  if entity = Client then
    ()
  else
    failwith "wrong sessid";
  let sess = getcSession sessid in
  let sh = sess.sh in 
  let csl = gen_csl sh.cipher_suite in
  let conn = initConn net_conn Client (ver2PV TLS1p0) in
  let (conn,ch,sh,buf,hsm) = clientHelloMsgs conn init_ver sess.sid csl in
  let cr = ch.ch_random in
  if sess.sid = sh.sh_session_id then
    let readst, writest = generateKeys Client cr sh sess.ms in
    let subj = x509subject (getCerts sess.server_cert) in
    storeState finResDb ("cKEresphase",cr) (conn, readst, writest, buf, hsm, subj, sess.pms, sess, ch, sh)
  else
    failwith "just for PV: only resumption with this function";
  cr


(* receive ClientHello *)
let serverCltHelloMsg conn = 
  let buf = empty_bstr in
  let conn,hsClientHello,ch,buf = recvClientHello conn buf in
  printHandshake hsClientHello;  
  printClientHello "" ch;
  (conn,ch,buf,hsClientHello)

(* send ServerHello *)
let serverSrvHelloMsg conn sh hsm = 
  let conn = updateConn_version conn sh.sh_session_id sh.server_version in
  print_msg "sending ServerHello\n";
  let conn,hsServerHello = sendHandshake_null conn HT_server_hello (HB_server_hello sh)  in
  let hsm = concat hsm hsServerHello in
  printHandshake hsServerHello;  
  printServerHello "" sh;
  (conn,hsm)


let serverHelloMsgsNoRes conn sver_min sver_max pref_csl = 
  let (conn,ch,buf,hsm) = serverCltHelloMsg conn in
  let (sh,session) = bldNewServerHello ch sver_min sver_max pref_csl in
  let (conn,hsm) = serverSrvHelloMsg conn sh hsm in
  (conn,ch,sh,buf,hsm,session)

let serverHelloMsgsRes conn sver_min sver_max pref_csl = 
  let (conn,ch,buf,hsm) = serverCltHelloMsg conn in
  let (sh,session) = bldResServerHello ch sver_min sver_max pref_csl in
  let (conn,hsm) = serverSrvHelloMsg conn sh hsm in
  (conn,ch,sh,buf,hsm,session)

let serverHelloMsgsTryRes conn sver_min sver_max pref_csl resume = 
  let (conn,ch,buf,hsm) = serverCltHelloMsg conn in
  let (sh,session) = bldTryResServerHello ch sver_min sver_max pref_csl resume in
  let (conn,hsm) = serverSrvHelloMsg conn sh hsm in
  (conn,ch,sh,buf,hsm,session)


(* no resumption *)
let serverHelloMsgs_att net_conn sver_min sver_max pref_csl = 
  let conn = initConn net_conn Server (ver2PV TLS1p0) in
  let resume = false in
  let (conn,ch,sh,buf,hsm,session) = serverHelloMsgsNoRes conn sver_min sver_max pref_csl in
  let sr = sh.sh_random in
  let cell = (conn,ch,sh,buf,hsm) in
  storeState hmDb ("sHMphase",sr) cell;
  sr


let serverHelloMsgsRes_att net_conn sver_min sver_max pref_csl = 
  let conn = initConn net_conn Server (ver2PV TLS1p0) in
#if fs
  let resume = true in
  let (conn,ch,sh,buf,hsm,session) = serverHelloMsgsTryRes conn sver_min sver_max pref_csl resume in
#else
  let (conn,ch,sh,buf,hsm,session) = serverHelloMsgsRes conn sver_min sver_max pref_csl in
#endif

  let sr = sh.sh_random in
  let () = 
    match session with
    | Some sess -> 
	begin
          print_msg "performing an abbreviated handshake\n\n";
	  let readst, writest = generateKeys Server ch.ch_random sh sess.ms in
	  let subj = x509subject (getCerts sess.server_cert) in
	  let cell = (conn, readst, writest, buf, hsm, subj, sess.pms, sess, ch, sh) in
	  storeState finResDb ("sKEresphase", sr) cell;
	end
    | None ->
#if fs
	  begin
	    print_msg "performing a full handshake\n\n";
	    storeState hmDb ("sHMphase", sr) (conn, ch, sh, buf, hsm)
	  end;
#else
          failwith "just for PV: only resumption with this function"
#endif
  in
  sr







let clientKeyExchange_phase conn buf hsm server_identity ch sh = 
  let client_version, ch_random,_,_,_,_  = decompClientHello ch in
  let server_version, sh_random,_,_,_ = decompServerHello sh in

  (* receive Certificate *)
  let conn, hsCertificate, certificate, buf = recvCertificate conn buf in
  printHandshake hsCertificate;
  let hsm = concat hsm hsCertificate in

  (* receive ServerHelloDone *)
  let conn,hsServerHelloDone,serverHelloDone,buf = recvServerHelloDone conn buf in
  printHandshake hsServerHelloDone;
  let hsm = concat hsm hsServerHelloDone in

  (* verify that the server provided a valid certificate 
     and check that the server hello parameters are acceptable *)
  // verify the server version, 
  // verify that the ciphersuite is in the list we sent (same for the compression method)

  (* verify the certificate *)
  let certs = getCerts certificate in
#if concrete
  ignore(Crypto.x509verify certs server_identity);
#else
  if certs = Prins.getX509Cert server_identity then
    ()
  else
    failwith "bad certificate";
#endif
  let public_key = asympubkey (x509pubkey certs) in

#if cv
#if thm_handshake_auth_comp
#else
  secret_nonce random;
#endif
#else
  let random = (mkRandom 46) in
#endif
  let pms = concat (bytes_of_ProtocolVersion client_version) (noncetobytes random) in
  let ver = conn.crt_version in
  let encpms = bldEncryptedPreMasterSecret ver public_key pms in
  let clientKeyExchange = { exchange_keys = CKEK_rsa encpms } in
  print_msg "sending ClientKeyExchange\n";
  let conn,hsClientKeyExchange = sendHandshake_null conn HT_client_key_exchange (HB_client_key_exchange clientKeyExchange) in
  let hsm = concat hsm hsClientKeyExchange in

  (* compute ms and keys *)
  let ms = compute_master_secret pms server_version ch_random sh_random in
  let readst, writest = generateKeys Client ch_random sh ms in
#if cv (* 8 *)
  conn, readst, writest, buf, hsm, certificate, empty_bstr, encpms
#else
  conn, readst, writest, buf, hsm, certificate, random, ms 
#endif (* 8 *)
(* Obs: we need to put random and not pms in the correspondance query
because PV fails to observe that if attacker:random then attacker:pms
*)

let clientKeyExchange_phase_att (cr:Random) server_identity = 
  let conn, ch, sh, buf, hsm = getState hmDb ("cHMphase",cr) in
  let conn, readst, writest, buf, hsm, cert, pms, ms = clientKeyExchange_phase conn buf hsm server_identity ch sh in
  let subj = x509subject (getCerts cert) in
  let sess = mkSession sh.sh_session_id cert pms ms ch sh in
  let ocell = (conn, readst, writest, buf, hsm, subj, pms, sess) in
  storeState finDb ("cKEphase",cr) ocell






let verifyPMS pms ver = 
  let b_ver = bytes_of_ProtocolVersion ver in
  let prefix,random = parsePMS pms in
  if prefix = b_ver then
    random
  else
#if concrete
    begin 
      printf "Client_version and PMS version do not correspond: %s vs "
	(string_of_ProtocolVersion ver);
      (print_raw prefix);
      printf "\n";
      failwith "client_version and PMS version do not correspond"
    end
#else
  failwith "client_version and PMS version do not correspond"
#endif 

(* concretely, a variant which should work for any version *)
let getPublicKeyEncrypted (version:ProtocolVersion) hlen msg = 
#if concrete
  let hlen = (int_of_bytes 3 hlen) in
  let len =
    try
      int_of_bytes 2 msg
    with TLSException_internal_error -> raise TLSException_decode_error
  in
  if len + 2 <> hlen then
    Bytearray.sub msg 0 hlen 
  else
    Bytearray.sub msg 2 len 
#else
  let len,encpms = iconcat msg in
  encpms
#endif




#if cv
let serverKeyExchange_phase conn buf hsm certName ch sh = 
()
#else
let serverKeyExchange_phase conn buf hsm certName ch sh = 
  (* send Certificate *)
  let x509 = Prins.getX509 certName in
  let certificate = bldCerts (Prins.cert x509) in 
  print_msg "sending Certificate\n";
  let conn,hsCertificate = sendHandshake_null conn HT_certificate (HB_certificate certificate) in
  let hsm = concat hsm hsCertificate in

  (* send ServerHelloDone *)
  let serverHelloDone = ServerHelloDoneCstr empty_bstr in
  print_msg "sending ServerHelloDone\n";
  let conn,hsServerHelloDone = sendHandshake_null conn HT_server_hello_done (HB_server_hello_done serverHelloDone)  in
  let hsm = concat hsm hsServerHelloDone in

  (* receive ClientKeyExchange *)
  let conn,hsClientKeyExchange,clientKeyExchange,buf,cke_len = recvClientKeyExchange conn buf in
  printHandshake hsClientKeyExchange;
  let hsm = concat hsm hsClientKeyExchange in

  let privateKey = Prins.privkey x509 in
  let encpms = match clientKeyExchange.exchange_keys with CKEK_rsa x -> x.pre_master_secret in

  let version = conn.crt_version in
  let ciphertext = getPublicKeyEncrypted version cke_len encpms in
  let pms = rsa_decrypt privateKey ciphertext in
  let random = verifyPMS pms ch.client_version in
  let cr = ch.ch_random in
  let ms = compute_master_secret pms sh.server_version cr sh.sh_random in
  let readst, writest = generateKeys Server cr sh ms in
  (conn, readst, writest, buf, hsm, certificate, random, ms)


let serverKeyExchange_phase_att (sr:Random) certName = 
  let conn, ch, sh, buf, hsm = getState hmDb ("sHMphase",sr) in
  let conn, readst, writest, buf, hsm, cert, pms, ms = serverKeyExchange_phase conn buf hsm certName ch sh in
  let subj = x509subject (getCerts cert) in
  let sess = mkSession sh.sh_session_id cert pms ms ch sh in
  let ocell = (conn, readst, writest, buf, hsm,  subj, pms, sess) in
  storeState finDb ("sKEphase",sr) ocell


#endif
(** sending the Finished message **)

let sendCCSbldFinished conn connst ms entity hsm =
  let ver = conn.crt_version in

  (* send ChangeCipherSpec message *)
  print_msg "sending ChangeCipherSpec\n";
#if cv (* 9 *)
#else
  let conn = sendccs conn ccs_msg in
  let conn = updateWriteConnectionState conn connst in
#endif (* 9 *)
  (* build Finished message *)
  let finished = { verify_data = bldVerifyData ver ms entity hsm } in
  print_msg "sending Finished\n";
  conn,finished




let getId sess = 
  let ch = sess.ch in
  let sh = sess.sh in
  concat ch.ch_random sh.sh_random

let sendFinished conn entity connst hsm subj pms sess =
  let conn, finished = sendCCSbldFinished conn connst sess.ms entity hsm in
  let id = getId sess in
  #if cv
    #if thm_handshake_auth_comp
      Pi.log tr (SendFinished(entity,sess.ms,conn));
    #endif
  #else
    Pi.log tr (SendFinished(entity,sess,conn));
  #endif

  #if cv
    let conn,hsFinished = sendHandshake_null conn HT_finished (HB_finished finished) in
  #else
    let conn,hsFinished = sendHandshake conn HT_finished (HB_finished finished) in
  #endif
  (conn, concat hsm hsFinished)

#if cv (* 10 *)

#if thm_handshake_auth_comp

let wrapper_sendFinished conn connst entity hsm ch sh subj cert =
  let client_version, ch_random,_,_,_,_  = decompClientHello ch in
  let server_version, sh_random,_,_,_ = decompServerHello sh in
  let pms = concat (bytes_of_ProtocolVersion client_version) (noncetobytes random) in
  let ms = compute_master_secret (noncetobytes random) empty_bstr ch_random sh_random in
  let sess = mkSession sh.sh_session_id cert pms ms ch sh in
  sendFinished conn entity connst hsm subj pms sess 

#else

let wrapperFinished conn connst ms entity hsm ch sh =
  let client_version, ch_random,_,_,_,_  = decompClientHello ch in
  let server_version, sh_random,_,_,_ = decompServerHello sh in
  let ms = compute_master_secret (noncetobytes random) empty_bstr ch_random sh_random in
  sendCCSbldFinished conn connst (noncetobytes ms) entity hsm

#endif

#else (* 10 *)

let sendFinishedRes conn entity connst hsm subj pms sess ch sh =
  let conn, finished = sendCCSbldFinished conn connst sess.ms entity hsm in
  let id = concat ch.ch_random sh.sh_random in
  Pi.log tr (SendFinishedRes(entity, sess, conn, ch, sh));
  let conn,hsFinished = sendHandshake conn HT_finished (HB_finished finished) in
  (conn, concat hsm hsFinished)


(* client: from sending the Finished message to receiving it *)
let cSendFinished_att (r:Random)  =
  let icell = getState finDb ("cKEphase",r) in
  let (conn, readst, writest, buf, hsm, subj, pms, sess) = icell in
  let conn, hsm = sendFinished conn Client writest hsm subj pms sess in
  let ocell = (conn, readst, writest, buf, hsm, subj, pms, sess) in
  storeState finDb ("cRecvFin",r) ocell


(* server: just sending the Finished message *)
let sSendFinished_att (r:Random) =
  let icell = getState finDb ("sSendFin",r)  in
  let (conn, readst, writest, buf, hsm,  subj, pms, sess) = icell in
  let conn, hsm = sendFinished conn Server writest hsm subj pms sess in
  let ch,sh,pms,ms = sess.ch,sess.sh,sess.pms,sess.ms in
  callCheckSecrecy Server sess conn;
  let connid = (concat ch.ch_random r, Server) in
  storeConnection connid conn;
  let sessid = (sh.sh_session_id, Server) in
  storesSession sessid sess;
  connid, sessid

(* version of sSendFinished_att for the password-based-client-auth application *)
let sSendFinishedPwd_att (r:Random) =
  let icell = getState finDb ("sSendFin",r)  in
  let (conn, readst, writest, buf, hsm,  subj, pms, sess) = icell in
  let conn, hsm = sendFinished conn Server writest hsm subj pms sess in
  let ch,sh,pms,ms = sess.ch,sess.sh,sess.pms,sess.ms in
  let connid = (concat ch.ch_random r, Server) in
  storeConnection connid conn;
  let sessid = (sh.sh_session_id, Server) in
  storesSession sessid sess;
  storeState pwdAppDb ("sPwdApp",connid) (conn,subj);
  connid, sessid


(* server resumption: from sending the Finished message to receiving it *)
let sSendFinishedRes_att (sr:Random) =
  let icell = getState finResDb ("sKEresphase",sr) in
  let conn, readst, writest, buf, hsm, subj, pms, sess, ch, sh = icell in
  let conn, hsm = sendFinishedRes conn Server writest hsm subj pms sess ch sh in
  let ocell = (conn, readst, writest, buf, hsm, subj, pms, sess, ch, sh) in
  storeState finResDb ("sRecvFinRes",sr) ocell


(* client resumption: just sending the Finished message *)
let cSendFinishedRes_att (cr:Random) =
  let icell = getState finResDb ("cSendFinRes",cr) in
  let conn, readst, writest, buf, hsm, subj, pms, sess, ch, sh = icell in
  let conn, hsm = sendFinishedRes conn Client writest hsm subj pms sess ch sh in
  let pms, ms = sess.pms, sess.ms in
  callCheckSecrecy Client sess conn;
  let connid = (concat cr sh.sh_random, Client) in
  storeConnection connid conn;
  let sessid = (sh.sh_session_id, Client) in
  connid, sessid


#endif (* 10 *)

(** receiving the Finished message **)

let peerEntity entity = 
  match entity with
  | Client -> Server
  | Server -> Client


let checkprf ver ms entity hsm finished = 
(* concretely we just check equality, computationally we call an equiv function *)
#if cv
  let tls_label = 
    match entity with
    | Client -> str "client finished"
    | Server -> str "server finished"
  in
  checkprf_hmac ms tls_label (concat (md5 hsm) (sha1 hsm)) 12 finished.verify_data
#else
  let verify_data = bldVerifyData ver ms (peerEntity entity) hsm in  
  verify_data = finished.verify_data
#endif

let recvFinished conn entity connst buf hsm ms =
  (* receive ChangeCipherSpec message *)
  let conn,msg = recvccs conn in
  if msg = ccs_msg then
    ()
  else
    failwith "decode error";
  let conn = updateReadConnectionState conn connst in
  
  (* receive Finished message *)
  let conn,hsFinished,finished,buf = recvHsFinished conn buf in
  printHandshake hsFinished;

  let ver = conn.crt_version in

  if checkprf ver ms entity hsm finished = true then 
    ()
  else
    failwith "decrypt error";
  (conn, buf, concat hsm hsFinished)


#if cv (* 11 *)

let wrapper_recvFinished conn connst entity hsm ch sh subj cert buf =
  let client_version, ch_random,_,_,_,_  = decompClientHello ch in
  let server_version, sh_random,_,_,_ = decompServerHello sh in
  let pms = concat (bytes_of_ProtocolVersion client_version) random in
  let ms = compute_master_secret (noncetobytes random) empty_bstr ch_random sh_random in
  let conn, buf, hsm = recvFinished conn entity connst buf hsm (noncetobytes ms) in
  let read, write = conn.read, conn.write in
  let sess = mkSession sh.sh_session_id cert pms ms ch sh in
  let id = getId sess in
    #if cv
      #if thm_handshake_auth_comp
        Pi.log tr (AcceptFinished(Client,noncetobytes ms,conn))
      #else
        ()
      #endif
    #else
      ()
    #endif

#else (* 11 *)

(* client: just receiving the Finished message *)
let cRecvFinished_att (r:Random)  =
  let icell = getState finDb ("cRecvFin",r) in
  let (conn, readst, writest, buf, hsm, subj, pms, sess) = icell in
  let ch,sh,ms = sess.ch,sess.sh,sess.ms in
  let conn, buf, hsm = recvFinished conn Client readst buf hsm ms in
  let read, write = conn.read, conn.write in
  let id = getId sess in
  Pi.log tr (AcceptFinished(Client,sess,conn));
  callCheckSecrecy Client sess conn;
  let connid = (id,Client) in
  storeConnection connid conn;
  let sessid = (sh.sh_session_id, Client) in
  storecSession sessid sess;
  connid, sessid

(* version of cRecvFinished_att for the password-based-client-auth application *)
let cRecvFinishedPwd_att (r:Random)  =
  let icell = getState finDb ("cRecvFin",r) in
  let (conn, readst, writest, buf, hsm, subj, pms, sess) = icell in
  let ch,sh,ms = sess.ch,sess.sh,sess.ms in
  let conn, buf, hsm = recvFinished conn Client readst buf hsm ms in
  let read, write = conn.read, conn.write in
  let id = getId sess in
  Pi.log tr (AcceptFinished(Client,sess,conn));
  let connid = (id,Client) in
  storeConnection connid conn;
  let sessid = (sh.sh_session_id, Client) in
  storecSession sessid sess;
  storeState pwdAppDb ("cPwdApp",connid) (conn,subj);
  connid, sessid


(* server: from receiving the Finished message to sending it *)
let sRecvFinished_att (r:Random) =
  let icell = getState finDb ("sKEphase",r) in
  let conn, readst, writest, buf, hsm, subj, pms, sess = icell in
  let conn, buf, hsm = recvFinished conn Server readst buf hsm sess.ms in
  let id = getId sess in
  Pi.log tr (AcceptFinished(Server,sess,conn));
  let ocell = (conn, readst, writest, buf, hsm, subj, pms, sess) in
  storeState finDb ("sSendFin",r) ocell


(* server resumption : just receiving the Finished message *)
let sRecvFinishedRes_att (sr:Random) =
  let conn, readst, writest, buf, hsm, subj, pms, sess, ch, sh = getState finResDb ("sRecvFinRes",sr)  in
  let pms,ms = sess.pms,sess.ms in
  let conn, buf, hsm = recvFinished conn Server readst buf hsm sess.ms in
  let read,write = conn.read,conn.write in
  let id = concat ch.ch_random sh.sh_random in
  Pi.log tr (AcceptFinishedRes(Server,sess,conn,ch,sh));
  callCheckSecrecy Server sess conn;
  let connid = (id,Server) in
  storeConnection connid conn;
  let sessid = (sh.sh_session_id, Server) in
  connid, sessid

(* client resumption : from receiving the Finished message to sending it *)
let cRecvFinishedRes_att (cr:Random) =
  let icell = getState finResDb ("cKEresphase",cr) in
  let conn, readst, writest, buf, hsm, subj, pms, sess, ch, sh = icell in
  let conn, buf, hsm = recvFinished conn Client readst buf hsm sess.ms in
  let id = concat ch.ch_random sh.sh_random in
  Pi.log tr (AcceptFinishedRes(Client,sess,conn,ch,sh));
  let ocell = (conn, readst, writest, buf, hsm, subj, pms, sess, ch, sh) in
  storeState finResDb ("cSendFinRes",cr) ocell




(* finalizing phase *)
let finalizeFullHS conn readst writest buf ch_random hsm subj pms sess = 
  let (conn,hsm) = sendFinished conn Client writest hsm subj pms sess in
  let (conn,buf,hsm) = recvFinished conn Client readst buf hsm sess.ms in
  print_msg "Full handshake completed successfully\n\n";
  let sh = sess.sh in
  let connid = (concat ch_random sh.sh_random, Client) in
  storeConnection connid conn;
  let sid = sh.sh_session_id in
  let sessid = (sid,Client) in
#if concrete
  if sid <> bytes_of_int 1 0 then
#endif
    storecSession sessid sess;
  connid, sessid


#endif (* 11 *)



(***** CLIENT *****)
#if cv (* AA *)
#else
let connect_ext (net_conn:Net.conn) minver maxver ccsl (server_identity:ServerName) : ConnectionId * SessionId = 
  let conn = initConn net_conn Client minver in
  let (conn,ch,sh,buf,hsm) = clientHelloMsgs conn maxver empty_sid ccsl in  
#if concrete (* not in the spec, only for testing *)
  if lt_ver conn.crt_version minver then
    failwith "The version selected by the server is not acceptable (too old).";
#endif
  let conn, readst, writest, buf, hsm, cert, pms, ms = clientKeyExchange_phase conn buf hsm server_identity ch sh in
  let subj = x509subject (getCerts cert) in
  let sess = mkSession sh.sh_session_id cert pms ms ch sh in
  finalizeFullHS conn readst writest buf ch.ch_random hsm subj pms sess

(* we don't give this to the attacker; we just use it for testing that the attacker *)
let connect_ext_att (net_conn:Net.conn) minver maxver ccsl (server_identity:str) : ConnectionId * SessionId = 
  let ch_random = clientHelloMsgs_att net_conn maxver empty_sid ccsl in  
  clientKeyExchange_phase_att ch_random server_identity;
  cSendFinished_att ch_random;
  cRecvFinished_att ch_random




(*** CLIENT RESUMPTION ***)

(* the first exchange should be almost identical, next we decide whether it is a resumption or not *)
let resume_ext (conn:Connection) (sess: Session) ccsl (ver: ProtocolVersion option): ConnectionId * SessionId = 
  print_msg "Trying to resume handshake\n";

  let sh = sess.sh in
  let init_ver = match ver with Some x -> x | None -> sh.server_version in 

  // we should send at least the established ciphersuite, ie verify that sess.cs is in ccsl
  let (conn,ch,sh,buf,hsm) = clientHelloMsgs conn init_ver sess.sid ccsl in

  let cr = ch.ch_random in

  if sess.sid = sh.sh_session_id then
    begin    (* resumption indeed *)
      (* update conn with new params and old ms, and compute the new keys *) 
      let subj = x509subject (getCerts sess.server_cert) in
      let readst, writest = generateKeys Client cr sh sess.ms in
      let (conn,buf,hsm) = recvFinished conn Client readst buf hsm sess.ms in
      let (conn,hsm) = sendFinished conn Client writest hsm subj sess.pms sess in      
      print_msg "Resumption completed successfully\n";
      let connid = (concat ch.ch_random sh.sh_random, Client) in
      storeConnection connid conn;      
      connid, (sess.sid, Client)
    end
  else
    begin    (* full handshake *)
      print_msg "The server has not accepted resumption for this session.\n\n";
      let server_identity = getCN (x509subject (getCerts sess.server_cert)) in
      let conn, readst, writest, buf, hsm, cert, pms, ms = clientKeyExchange_phase conn buf hsm server_identity ch sh in
      let subj = x509subject (getCerts cert) in
      let sess = mkSession sh.sh_session_id cert pms ms ch sh in
      finalizeFullHS conn readst writest buf cr hsm subj pms sess
    end



(* we assume here that all stored sessions are resumable *)
let resume_ext_att (net_conn:Net.conn) (sessid: SessionId) ccsl : ConnectionId * SessionId = 
  print_msg "Trying to resume handshake\n";
  let cr = clientHelloMsgsRes_att net_conn sessid (ver2PV TLS1p0) in
  cRecvFinishedRes_att cr;
  cSendFinishedRes_att cr
(* full handshake if session is not found or is not resumable: here we don't do it *)
(* let sess = getSession sessid in
   let server_identity = getCN (x509subject (getCerts sess.server_cert)) in
   clientKeyExchange_phase_att cr server_identity;
   cSendFinished_att cr;
   cRecvFinished_att cr
*)


  


(***** SERVER *****)

let serverFullHS conn buf hsm certName ch sh = 
  print_msg "performing a full handshake\n\n";
  let conn, readst, writest, buf, hsm, cert, pms, ms = serverKeyExchange_phase conn buf hsm certName ch sh in
  let subj = x509subject (getCerts cert) in
  let sess = mkSession sh.sh_session_id cert pms ms ch sh in
  let readst, writest = generateKeys Server ch.ch_random sh ms in
  let (conn,buf,hsm) = recvFinished conn Server readst buf hsm ms in
  let (conn,hsm) = sendFinished conn Server writest hsm subj pms sess in
  print_msg "Full handshake completed successfully\n\n";
  conn,cert,pms,ms
    
let serverAbbrvHS conn buf hsm certName ch sh session = 
  print_msg "performing an abbreviated handshake\n\n";
  let cert = session.server_cert in
  let subj = x509subject (getCerts cert) in
  let ms = session.ms in
  let readst, writest = generateKeys Server ch.ch_random sh ms in
  let (conn,hsm) = sendFinished conn Server writest hsm subj session.pms session in
  let (conn,buf,hsm) = recvFinished conn Server readst buf hsm ms in
  print_msg "Abbreviated handshake completed successfully\n\n";
  conn,cert,session.pms,ms

let accept_ext (net_conn:Net.conn) (certName:CertName) (sver_min:ProtocolVersion) (sver_max:ProtocolVersion) (pref_csl:CipherSuites) : ConnectionId * SessionId = 
  print_msg "Handshake started\n";
  (* initialise connection: with TLS_NULL_NULL_NULL *)
  let conn = initConn net_conn Server (ver2PV TLS1p0) in

  let resume = true in 
  let (conn,ch,sh,buf,hsm,session) = serverHelloMsgsTryRes conn sver_min sver_max pref_csl resume in

  let conn,certificate,pms,ms = 
    match session with
    | Some sess -> serverAbbrvHS conn buf hsm certName ch sh sess
    | None -> serverFullHS conn buf hsm certName ch sh
  in
  
  (* storing the new connection and, if necessary, the new session *)
  let connid = (concat ch.ch_random sh.sh_random, Server) in
  storeConnection connid conn;

  let sid = sh.sh_session_id in
  let sessid = (sid,Server) in

  let () = match session with
  | Some sess -> ()
  | None -> 
      let sess = mkSession sid certificate pms ms ch sh in
      storesSession sessid sess
  in
  connid, sessid




#if fs
(* testing that the code corresponding to the attacker interface is executable *)
let accept_ext_att net_conn certName sver_min sver_max pref_csl = 
  print_msg "Handshake started\n";
  let sr = serverHelloMsgsRes_att net_conn sver_min sver_max pref_csl in
  try (
    serverKeyExchange_phase_att sr certName;
    sRecvFinished_att sr;
    let cid,sid = sSendFinished_att sr in
    print_msg "\nFull handshake completed successfully\n\n";
    cid, sid
   )
  with exn -> (
    sSendFinishedRes_att sr;
    let cid,sid = sRecvFinishedRes_att sr in
    print_msg "\nAbbreviated handshake completed successfully\n\n";
    cid,sid
   )

#endif 



(*************************************************************)

#endif (* AA *)




#if concrete
let ccslist = 
    [ TLS_RSA_WITH_AES_128_CBC_SHA;
      TLS_RSA_WITH_RC4_128_MD5;
      TLS_RSA_WITH_RC4_128_SHA;       
      TLS_RSA_WITH_DES_CBC_SHA;
      TLS_NULL_WITH_NULL_NULL;
      TLS_RSA_WITH_NULL_MD5;               
      TLS_RSA_WITH_NULL_SHA;
    ]
    
#else
let ccslist = 
      (bytes_of_CipherSuite TLS_RSA_WITH_AES_128_CBC_SHA, bytes_of_CipherSuite TLS_NULL_WITH_NULL_NULL)
#endif

#if fs
let connect (net_conn:Net.conn) server_identity : (ConnectionId*SessionId) = 
  connect_ext_att net_conn 
    (ver2PV TLS1p0)
    (ver2PV TLS1p1)
    ccslist
    server_identity

let rehandshake (connid: ConnectionId) (sessid: SessionId) : ConnectionId * SessionId = 
  let sess = getcSession sessid in
  let conn = getConnection connid in
  resume_ext conn sess ccslist None

let resume (net_conn:Net.conn) (sessid: SessionId) : ConnectionId * SessionId = 
  //let sess = getSession sessid in
  //let conn = initConn net_conn Client sess.init_version in
  //resume_ext conn sessid ccslist None
  resume_ext_att net_conn sessid ccslist

let accept (net_conn:Net.conn) (certName:CertName) : ConnectionId * SessionId = 
  accept_ext_att net_conn certName (ver2PV TLS1p0) (ver2PV TLS1p1) ccslist
#endif


#if cv (* BB *)
#else
let close (connid:ConnectionId) (wait_resp:bool) = 
  let conn = getConnection connid in
  let close_alert = bytes_of_AlertDescription AD_close_notify in
  print_msg "sending CloseNotify alert\n\n";
  let conn = send_alert conn close_alert in
  (* we might receive a corresponding close alert *)
  (* how we should handle this case? see 7.2.1.in rfc2246 *)
  (* it depends on the application whether we wait or not *)
  if wait_resp then
    let conn,msg = recv_alert conn in
    // verify it is the right one? no, because we stop anyhow
    ()
  else
    ()


 

let generateConnection () : ConnectionId * ConnectionId =
  failwith "use the files 'recordonly.ml[i]'"

#endif (* BB *)
