module Formats

open Data
open Formats_ciphersuites


(* The types defined here follow as much as possible the the protocol types described in Appendix A of the RFC2246 *)
(* The protocol constants are defined in the constants.ml file *)


(* A.1. Record layer *)

type version = bytes // 2 bytes

val TLS1p0: version
val TLS1p1: version
#if concrete
val SSLv2: version
val SSLv3: version
val TLS1p2: version
#endif

#if concrete
type ProtocolVersion = { major: int; minor: int }
#else
(* when not doing pattern matching; useful for obtaining more
efficient ProVerif reductions *)
type ProtocolVersion = version
#endif

(* for pattern matching *)
type pm_version = 
    | PM_TLS1p0
    | PM_TLS1p1
#if concrete
    | PM_SSLv2
    | PM_SSLv3
    | PM_TLS1p2
#endif


type ContentType = bytes
val CT_change_cipher_spec: ContentType
val CT_alert: ContentType
val CT_handshake: ContentType
val CT_application_data: ContentType

 
type 'a publickeyencrypted = bytes
type 'a digitallysigned = bytes



(* A.2. Change cipher specs message *)

type ChangeCipherSpec = CCS_change_cipher_spec


(* A.3. Alert messages *)


type AlertLevel = 
    | AL_warning
    | AL_fatal
    | AL_unknown_level of int

type AlertDescription = 
    | AD_close_notify		
    | AD_unexpected_message	
    | AD_bad_record_mac		
    | AD_decryption_failed
    | AD_record_overflow	
    | AD_decompression_failure
    | AD_handshake_failure	
    | AD_no_certificate		
    | AD_bad_certificate		
    | AD_unsupported_certificate	
    | AD_certificate_revoked	
    | AD_certificate_expired	
    | AD_certificate_unknown	
    | AD_illegal_parameter	
    | AD_unknown_ca		
    | AD_access_denied		
    | AD_decode_error		
    | AD_decrypt_error		
    | AD_export_restriction	
    | AD_protocol_version	
    | AD_insufficient_security	
    | AD_internal_error		
    | AD_user_cancelled		
    | AD_no_renegotiation	
    | AD_unknown_description of int

type Alert = {level: AlertLevel; description: AlertDescription }

#if concrete
exception TLSException_decode_error
exception TLSException_internal_error
exception TLSException_unexpected_message
exception TLSException_decrypt_error
exception TLSException_decryption_failed
exception TLSException_bad_record_mac
exception TLSException_handshake_failure
exception TLSexception_bad_certificate
exception TLSException_protocol_version
exception TLSException_not_yet_implemented
exception TLSException_empty_message
#endif


(* A.4. Handshake protocol *)

(* A.4.1. Hello messages *)

type HelloRequest = HelloRequestCstr

type Random = bytes
type SessionID = bytes
type bCipherSuite = bytes
type CompressionMethod = CM_null  
#if concrete
type CipherSuites = CipherSuite list
type CompressionMethods = CompressionMethod list 
#else
type CipherSuites = bCipherSuite * bCipherSuite
type CompressionMethods = CompressionMethod
#endif
val cml_null: CompressionMethods

type ClientHello = {
    client_version: ProtocolVersion;
    ch_random: Random;
    ch_session_id: SessionID;
    cipher_suites: CipherSuites;
    compression_methods: CompressionMethods;
    extra_data: bytes;
  }

type ServerHello = {
    server_version: ProtocolVersion;
    sh_random: Random;
    sh_session_id: SessionID;
    cipher_suite: bCipherSuite;
    compression_method: CompressionMethod;
  }


(* A.4.2. Server authentication and key exchange messages *)

type ASN1Cert = bytes
type Certificate = { certificate_list: ASN1Cert }
type ServerHelloDone = ServerHelloDoneCstr of bytes


(* A.4.3. Client authentication and key exchange messages *)

type PreMasterSecret = {
    pms_client_version: ProtocolVersion;
    pms_random: bytes;
  }
type EncryptedPreMasterSecret = { pre_master_secret: PreMasterSecret publickeyencrypted }
type ClientKeyExchangeKeys = CKEK_rsa of EncryptedPreMasterSecret
type ClientKeyExchange = { exchange_keys: ClientKeyExchangeKeys } 



(* A.4.4. Handshake finalization message *)

type Finished = { verify_data: bytes }


(* A.4. Handshake protocol *)

type HandshakeType = bytes
val HT_hello_request: HandshakeType
val HT_client_hello: HandshakeType
val HT_server_hello: HandshakeType
val HT_certificate: HandshakeType
val HT_server_key_exchange: HandshakeType
val HT_certificate_request: HandshakeType
val HT_server_hello_done: HandshakeType
val HT_certificate_verify: HandshakeType
val HT_client_key_exchange: HandshakeType
val HT_finished: HandshakeType


type HandshakeBody = 
  | HB_client_hello of         ClientHello
  | HB_server_hello of         ServerHello
  | HB_certificate of          Certificate
  | HB_server_hello_done of    ServerHelloDone
  | HB_client_key_exchange of  ClientKeyExchange
  | HB_finished of             Finished

type Handshake = {
  msg_type: HandshakeType;
  length: int;
  body: HandshakeBody;
  }

(* A.5. The CipherSuite *)
(* see the Formats_ciphersuites module *)

(* A.6. The Security Parameters *)

(*--- formatConnEndBegin *)
type ConnectionEnd = Client | Server
(*--- formatConnEndEnd *)
type BulkCipherAlgorithm = BCA_rc4 | BCA_des | BCA_aes | BCA_rc2 | BCA_3des | BCA_des40 | BCA_idea | BCA_null
type CipherType = CT_stream | CT_block     
type IsExportable = bool
type MACAlgorithm = MA_md5 | MA_sha1 | MA_null


(* We only store here the negotiated parameters which are used by the Record Protocol *)
(*--- paramsBegin *)
type SecurityParameters = {
    cipher_type: CipherType;
    bulk_cipher_algorithm: BulkCipherAlgorithm;
    mac_algorithm: MACAlgorithm; }
(*--- paramsEnd *)

(* The following values appear in the RFC's SecurityParameters, but
only the above three are need for functional correctness. Some of them
values below are recorded in the Session record. *)

(* The security parameters, as they appear in the RFC:
type SecurityParameters = {
    entity: ConnectionEnd;
    bulk_cipher_algorithm: BulkCipherAlgorithm;
    cipher_type: CipherType;
    key_size: int;
    key_material_length: int;
    is_exportable: IsExportable;
    mac_algorithm: MACAlgorithm;
    hash_size: int;
    compression_algorithm: CompressionMethod;
    master_secret: bytes;
    client_random: bytes;
    server_random: bytes
  }
*)

