module Formats

open Data
open Formats_ciphersuites


(* A.1. Record layer *)


type version = bytes
#if concrete
let SSLv2 = bytes_of_intpair (0,2)
let SSLv3 = bytes_of_intpair (3,0)
let TLS1p0 = bytes_of_intpair (3,1)
let TLS1p1 = bytes_of_intpair (3,2)
let TLS1p2 = bytes_of_intpair (3,3)
#else
//let SSLv2 = Utf8 (str "SSLv2")
//let SSLv3 = Utf8 (str "SSLv3")
let TLS1p0 = Utf8 (str "TLS1p0")
let TLS1p1 = Utf8 (str "TLS1p1")
//let TLS1p2 = Utf8 (str "TLS1p2")
#endif

#if concrete
type ProtocolVersion = { major: int; minor: int }
#else
type ProtocolVersion = version
#endif


type pm_version = 
    | PM_TLS1p0
    | PM_TLS1p1
#if concrete
    | PM_SSLv2
    | PM_SSLv3
    | PM_TLS1p2
#endif



type ContentType = bytes
#if concrete
let CT_change_cipher_spec = bytes_of_int 1 20
let CT_alert = bytes_of_int 1 21
let CT_handshake = bytes_of_int 1 22
let CT_application_data = bytes_of_int 1 23
#else
let CT_change_cipher_spec = Utf8 (str "change_cipher_spec")
let CT_alert = Utf8 (str "alert")
let CT_handshake = Utf8 (str "handshake")
let CT_application_data = Utf8 (str "application_data")
#endif


type 'a publickeyencrypted = bytes
type 'a digitallysigned = bytes


(* A.2. Change cipher specs message *)

type ChangeCipherSpec = CCS_change_cipher_spec 


(* A.3. Alert messages *)

type AlertLevel = 
    | AL_warning
    | AL_fatal
    | AL_unknown_level of int

type AlertDescription = 
    | AD_close_notify		
    | AD_unexpected_message	
    | AD_bad_record_mac		
    | AD_decryption_failed
    | AD_record_overflow	
    | AD_decompression_failure
    | AD_handshake_failure	
    | AD_no_certificate		
    | AD_bad_certificate		
    | AD_unsupported_certificate	
    | AD_certificate_revoked	
    | AD_certificate_expired	
    | AD_certificate_unknown	
    | AD_illegal_parameter	
    | AD_unknown_ca		
    | AD_access_denied		
    | AD_decode_error		
    | AD_decrypt_error		
    | AD_export_restriction	
    | AD_protocol_version	
    | AD_insufficient_security	
    | AD_internal_error		
    | AD_user_cancelled		
    | AD_no_renegotiation	
    | AD_unknown_description of int

type Alert = {level: AlertLevel; description: AlertDescription }

#if concrete
exception TLSException_decode_error
exception TLSException_internal_error
exception TLSException_unexpected_message
exception TLSException_decrypt_error
exception TLSException_decryption_failed
exception TLSException_bad_record_mac
exception TLSException_handshake_failure
exception TLSexception_bad_certificate
exception TLSException_protocol_version
exception TLSException_not_yet_implemented
exception TLSException_empty_message
#endif

(* A.4. Handshake protocol *)

(* A.4.1. Hello messages *)

type HelloRequest = HelloRequestCstr

type Random = bytes
type SessionID = bytes
type bCipherSuite = bytes
type CompressionMethod = CM_null
#if concrete
type CipherSuites = CipherSuite list
type CompressionMethods = CompressionMethod list 
let cml_null = [CM_null]
#else
type CipherSuites = bCipherSuite * bCipherSuite
type CompressionMethods = CompressionMethod
let cml_null = CM_null
#endif


type ClientHello = {
    client_version: ProtocolVersion;
    ch_random: Random;
    ch_session_id: SessionID;
    cipher_suites: CipherSuites;
    compression_methods: CompressionMethods;
    extra_data: bytes;
  }

type ServerHello = {
    server_version: ProtocolVersion;
    sh_random: Random;
    sh_session_id: SessionID;
    cipher_suite: bCipherSuite;
    compression_method: CompressionMethod;
  }


(* A.4.2. Server authentication and key exchange messages *)

type ASN1Cert = bytes
type Certificate = { certificate_list: ASN1Cert }

type KeyExchangeAlgorithm = KEA_rsa | KEA_diffie_hellman
type SignatureAlgorithm = SA_anonymous | SA_rsa | SA_dsa 
type RSAHash = {md5_hash: bytes; sha_hash: bytes}

type Signature = 
  | S_anonymous 
  | S_rsa of RSAHash digitallysigned

type ClientCertificateType = CCT_rsa_sign | CCT_dss_sign | CCT_rsa_fixed_dh | CCT_dss_fixed_dh
type DistinguishedName = bytes
type CertificateRequest = {
    certificate_types: ClientCertificateType;
    certificate_authorities: DistinguishedName;
  }

type ServerHelloDone = ServerHelloDoneCstr of bytes


(* A.4.3. Client authentication and key exchange messages *)

type PreMasterSecret = {
    pms_client_version: ProtocolVersion;
    pms_random: bytes;
  }
type EncryptedPreMasterSecret = { pre_master_secret: PreMasterSecret publickeyencrypted }
type ClientKeyExchangeKeys = CKEK_rsa of EncryptedPreMasterSecret
type ClientKeyExchange = { exchange_keys: ClientKeyExchangeKeys } 

type CertificateVerify = { signature: Signature }


(* A.4.4. Handshake finalization message *)

type Finished = { verify_data: bytes }


(* A.4. Handshake protocol *)

type HandshakeType = bytes
#if concrete
let HT_hello_request = bytes_of_int 1 0
let HT_client_hello = bytes_of_int 1 1
let HT_server_hello = bytes_of_int 1 2
let HT_certificate = bytes_of_int 1 11
let HT_server_key_exchange = bytes_of_int 1 12
let HT_certificate_request = bytes_of_int 1 13
let HT_server_hello_done = bytes_of_int 1 14
let HT_certificate_verify = bytes_of_int 1 15
let HT_client_key_exchange = bytes_of_int 1 16
let HT_finished = bytes_of_int 1 20
#else
let HT_hello_request = Utf8 (str "hello_request")
let HT_client_hello = Utf8 (str "client_hello")
let HT_server_hello = Utf8 (str "server_hello")
let HT_certificate = Utf8 (str "certificate")
let HT_server_key_exchange = Utf8 (str "server_key_exchange")
let HT_certificate_request = Utf8 (str "certificate_request")
let HT_server_hello_done = Utf8 (str "server_hello_done")
let HT_certificate_verify = Utf8 (str "certificate_verify")
let HT_client_key_exchange = Utf8 (str "client_key_exchange")
let HT_finished = Utf8 (str "finished")
#endif

type HandshakeBody = 
  | HB_client_hello of         ClientHello
  | HB_server_hello of         ServerHello
  | HB_certificate of          Certificate
  | HB_server_hello_done of    ServerHelloDone
  | HB_client_key_exchange of  ClientKeyExchange
  | HB_finished of             Finished

type Handshake = {
  msg_type: HandshakeType;
  length: int;
  body: HandshakeBody;
  }

(* A.6. The Security Parameters *)

type ConnectionEnd = Client | Server
type BulkCipherAlgorithm = BCA_rc4 | BCA_des | BCA_aes | BCA_rc2 | BCA_3des | BCA_des40 | BCA_idea | BCA_null
type CipherType = CT_stream | CT_block     
type IsExportable = bool
type MACAlgorithm = MA_md5 | MA_sha1 | MA_null


type SecurityParameters = {
    cipher_type: CipherType;
    bulk_cipher_algorithm: BulkCipherAlgorithm;
    mac_algorithm: MACAlgorithm;
  }
      
(* as in the spec:
type SecurityParameters = {
    entity: ConnectionEnd;
    bulk_cipher_algorithm: BulkCipherAlgorithm;
    cipher_type: CipherType;
    key_size: int;
    key_material_length: int;
    is_exportable: IsExportable;
    mac_algorithm: MACAlgorithm;
    hash_size: int;
    compression_algorithm: CompressionMethod;
    master_secret: bytes;
    client_random: bytes;
    server_random: bytes
  }
*)




