module Conversions

open Data
open Formats_ciphersuites
open Formats
open Conversions_lowlevel


(*** pretty-printing functions ***)

let string_of_version ver = 
#if concrete
  if ver = SSLv2 then "SSLv2"
  else if ver = SSLv3 then "SSLv3" 
  else if ver = TLS1p0 then "TLS1p0" 
  else if ver = TLS1p1 then "TLS1p1" 
  else if ver = TLS1p2 then "TLS1p2" 
  else failwith "unknown version"
#else
  match ver with
  | Utf8 (Literal x) -> x
  | _ -> failwith "bad version representation"
#endif

let string_of_ProtocolVersion pv = string_of_version (PV2ver pv)

let string_of_bCipherSuite bcs = 
  string_of_CipherSuite (CipherSuite_of_bytes bcs)

let string_of_ContentType ct = 
 if ct = CT_change_cipher_spec then "ChangeCipherSpec"
 else if ct= CT_alert then "Alert"
 else if ct= CT_handshake then "Handshake"
 else if ct = CT_application_data then "Application"
 else failwith "wrong parameter"




let string_of_CompressionMethod cm =
  match cm with
    CM_null -> "null"


let string_of_HandshakeType ht =
  if ht = HT_hello_request then "HelloRequest"
  else if ht = HT_client_hello then "ClientHello"
  else if ht = HT_server_hello then "ServerHello"
  else if ht = HT_certificate then "Certificate"
  else if ht = HT_server_key_exchange then "ServerKeyExchange"
  else if ht = HT_certificate_request then "CertificateRequest"
  else if ht = HT_server_hello_done then "ServerHelloDone"
  else if ht = HT_certificate_verify then "CertificateVerify"
  else if ht = HT_client_key_exchange then "ClientKeyExchange"
  else if ht = HT_finished then "Finished"
  else failwith "wrong handshake type"



(*** Version ***)

let bytes_of_ProtocolVersion (ver: ProtocolVersion) : bytes = 
  bytes_of_version (PV2ver ver)

let ProtocolVersion_of_bytes (msg: bytes) : ProtocolVersion = 
  ver2PV (version_of_bytes msg)



(*** Handshake messages ***)

let bytes_of_ClientHello (ch: ClientHello) : bytes = 
  concat6 
    (bytes_of_ProtocolVersion ch.client_version)
    ch.ch_random
    ch.ch_session_id
    (bytes_of_CipherSuites ch.cipher_suites)
    (bytes_of_CompressionMethods ch.compression_methods)
    ch.extra_data

let ClientHello_of_bytes (msg: bytes) : ClientHello = 
  let (b_ver, cr, sid, b_csl, b_cml, extra) = parseClientHello msg in
    { client_version = ProtocolVersion_of_bytes b_ver;
      ch_random = cr;
      ch_session_id = sid;
      cipher_suites = CipherSuites_of_bytes b_csl;
      compression_methods = CompressionMethods_of_bytes b_cml;
      extra_data = extra;
    }


let bytes_of_ServerHello (sh: ServerHello) : bytes = 
  concat5 
    (bytes_of_ProtocolVersion sh.server_version)
    sh.sh_random
    sh.sh_session_id
    sh.cipher_suite
    (bytes_of_CompressionMethod sh.compression_method)

let ServerHello_of_bytes (msg: bytes) : ServerHello = 
  let (b_ver, sr, sid, b_cs, b_cm) = parseServerHello msg in
    { server_version = ProtocolVersion_of_bytes b_ver;
      sh_random = sr;
      sh_session_id = sid;
      cipher_suite = b_cs;
      compression_method = CompressionMethod_of_bytes b_cm;
    }


let bytes_of_Certificate (cert: Certificate) : bytes = 
  cert.certificate_list 

let Certificate_of_bytes (cert_list: bytes) : Certificate = 
    { certificate_list = cert_list }


let bytes_of_ServerHelloDone (shd: ServerHelloDone) : bytes = empty_bstr

let ServerHelloDone_of_bytes (msg: bytes) = 
  if msg = empty_bstr then
    ServerHelloDoneCstr empty_bstr
  else
    failwith "decode error"


let bytes_of_ClientKeyExchange (cke: ClientKeyExchange) : bytes = 
  let (CKEK_rsa encpms) = cke.exchange_keys in
    encpms.pre_master_secret      

let ClientKeyExchange_of_bytes (msg: bytes) : ClientKeyExchange =
  let encpms = { pre_master_secret = msg } in
    { exchange_keys = CKEK_rsa encpms } 


let bytes_of_Finished (fin: Finished) : bytes = fin.verify_data

let Finished_of_bytes (msg: bytes) : Finished = { verify_data = msg }


 
(*** handshake <-> bytes ***)


let Handshake_of_bytes (hsmsg:bytes) : Handshake = 
  let (b_msg_type, b_length, b_body) = parseHandshake hsmsg in
  let len = int_of_bytes 3 b_length in
  let ht = HandshakeType_of_bytes (b_msg_type) in
  { 
    msg_type = ht;
    length = len;
    body = 
      if ht = HT_client_hello then HB_client_hello (ClientHello_of_bytes b_body)
      else if ht = HT_server_hello then HB_server_hello (ServerHello_of_bytes b_body)
      else if ht = HT_certificate then HB_certificate (Certificate_of_bytes b_body)
      else if ht = HT_server_hello_done then HB_server_hello_done (ServerHelloDone_of_bytes b_body)
      else if ht = HT_client_key_exchange then HB_client_key_exchange (ClientKeyExchange_of_bytes b_body)
      else if ht = HT_finished then HB_finished (Finished_of_bytes b_body)
      else 
	begin
	  printf "[Handshake_of_bytes] msg_type = %s\n" (string_of_HandshakeType ht);
	  failwith "not yet implemented"
	end 
  }


let bytes_of_Handshake (hs: Handshake) : bytes = 
  let b_body = 
    match hs.body with
      | HB_client_hello x -> (bytes_of_ClientHello x)
      | HB_server_hello x -> (bytes_of_ServerHello x)
      | HB_certificate x -> (bytes_of_Certificate x)
      | HB_server_hello_done x -> (bytes_of_ServerHelloDone x)
      | HB_client_key_exchange x -> (bytes_of_ClientKeyExchange x)
      | HB_finished x -> (bytes_of_Finished x)
  in
  let len = length b_body in 
    concat3 (bytes_of_HandshakeType hs.msg_type) (bytes_of_int 3 len) b_body






(*** other pretty-printing functions ***)

#if fs
let debug = ref true
let set_debug d = debug := d
#endif


#if fs
let print_tls ct ver len = 
  if !debug then
    Printf.printf "Content-type: %s\nVersion: %s\nLength: %d\n\n"  
      (string_of_ContentType ct) 
      (string_of_version (PV2ver ver))
      len 
#else
let print_tls ct ver len = ()
#endif

#if concrete
let print_hello str ver random sid csl_str cml_str = 
  if !debug then
    begin
      Printf.printf "%sHello\n" str;
      Printf.printf "Version: %s\nRandom (%d): "  
	(string_of_ProtocolVersion ver)
	(Bytearray.length random);
      print_raw random;
      Printf.printf "\nSession id (1+%d): " 
	(Byte.to_int (Bytearray.get sid 0));
      print_raw sid;
      Printf.printf "\nCiphersuites: \n\t%s\n" csl_str;
      Printf.printf "Compression methods: %s\n" cml_str
    end
	
let printClientHello str ch = 
  if !debug then
    begin
      let str = Printf.sprintf "%sClient" str in
      print_hello str ch.client_version ch.ch_random ch.ch_session_id
	(String.concat "\n\t" (List.map string_of_CipherSuite ch.cipher_suites))
	(String.concat "\t" (List.map string_of_CompressionMethod ch.compression_methods));
      print_bytes "Extra data" ch.extra_data;
      print_newline()
    end
	
let printServerHello str sh = 
  if !debug then
    begin
      let str = Printf.sprintf "%sServer" str in
      print_hello str sh.server_version sh.sh_random sh.sh_session_id
	(string_of_bCipherSuite sh.cipher_suite)
	(string_of_CompressionMethod sh.compression_method);
      print_newline()
    end
	
let printHandshake hs = 
  if !debug then
    let hs = Handshake_of_bytes hs in
    Printf.printf "Handshake type: %s\nHandshake length: %d \n\n"  
      (string_of_HandshakeType hs.msg_type) 
      hs.length 
#else
#if fs
let printClientHello str (ch:ClientHello) = 
  if !debug then
    begin
      printf "%s\n" str;
      print_any ch;
      printf "\n\n"
    end

let printServerHello str (sh:ServerHello) = 
  if !debug then
    begin
      printf "%s\n" str;
      print_any sh;
      printf "\n\n"
    end

let printHandshake (hs:bytes) = 
  if !debug then
    begin print_any hs; flush stdout end
#else
let printClientHello s ch = printf ""
let printServerHello s sh = printf ""
let printHandshake hs = printf "" 
#endif
#endif


