module App_http

open Formats_ciphersuites
open Formats
open Conversions

type URI = string

let getContentLength header = 
  let rec scan_lines lines = 
    match lines with
      [] -> 0
    | (line:string)::t -> 
	let len = line.Length in
	if (16 < len) && ((String.sub line 0 16) = "Content-Length: ") then
	  int_of_string (String.sub line 16 (len-16))
	else
	  scan_lines t
  in scan_lines (String.split ['\n'] header)

let getHeaderLength (frag:string) = 
  let prev = ref false in
  let len = frag.Length in
  let hlen = ref len in
  let test i =   
    (i<len && frag.[i] = '\r') &&
      (i+1<len && frag.[i+1] = '\n') &&
	(i+2<len && frag.[i+2] = '\r') &&
	  (i+3<len && frag.[i+3] = '\n')
  in	      
  for i = len - 1 downto 0 do
    if test i then
	hlen := i
  done;
  !hlen

let buildHttpGet uri = 
  let host = Net.get_host uri in
  let abspath = Net.get_abspath uri in
  let req = Printf.sprintf "GET %s HTTP/1.1\r\nHost: %s\r\n\r\n\r\n" 
            abspath host in
  Data.utf8 req

let parseHttpGetResponse resp connid = 
  let strfrag = Data.iutf8 resp in
  let content_len = getContentLength strfrag in
  let header_len = getHeaderLength strfrag in
  let rec recv_content current_len current_content =   
  (* approximation: we assume that we miscomputed the header_len,
     rather than expect a fragment containing less than 16 bytes *)
    if content_len - current_len >= 16 then
      begin
	let frag = Record.recv connid in
	let strfrag = Data.iutf8 frag in
	let currentcontent = current_content ^ strfrag in
	let currentlen = current_len + strfrag.Length in
	recv_content currentlen currentcontent
      end
    else
      current_content
  in
  let resp = strfrag ^ (recv_content (strfrag.Length - header_len) "") in
  Printf.printf "%s\n" resp

let recv_resp connid = 
  let frag = Record.recv connid in
  let strfrag = Data.iutf8 frag in
  let content_len = getContentLength strfrag in
  let header_len = getHeaderLength strfrag in
  let rec recv_content current_len current_content =   
    if content_len - current_len > 4 then
      begin
	let frag = Record.recv connid in
	let strfrag = Data.iutf8 frag in
	let currentcontent = current_content ^ strfrag in
	let currentlen = current_len + strfrag.Length in
	recv_content currentlen currentcontent
      end
    else
      current_content
  in
  strfrag ^ (recv_content (strfrag.Length - header_len) "")

let clientHttpSendReq connid uri = 
  let req = buildHttpGet uri in
  Record.send connid req

let clientHttpRecvReq connid = 
  let resp = recv_resp connid in
  Printf.printf "%s\n" resp


(*--- tlsconnectBegin *)
let tlsconnect uri = 
  let conn = Net.connect uri in 
  let shost = Net.get_host uri in 
  Handshake.connect conn shost
(*--- tlsconnectEnd *)

(*--- httpGetBegin *)
let httpGet connid uri = 
  let req = buildHttpGet uri in
  Record.send connid req;
  let resp = Record.recv connid in
  parseHttpGetResponse resp connid 
(*--- httpGetEnd *)

(*--- clientHttpsBegin *)
let clientHttps uri = 
  let connid, sessionid = tlsconnect uri in
  httpGet connid uri
(*--- clientHttpsEnd *)





let htmlize str = 
  let f (str:string) i = 
    str.Insert(i,"<br>"), i+5
  in
  let rec g (str:string) j =
    let i = ref j in 
    while !i < str.Length && str.[!i] <> '\n' do
      i := !i + 1
    done;
    if !i < str.Length then
      let str, jj = f str !i in
      g str jj
    else
      str
  in
  g str 0



let getSessionInfo sid = 
  let sess = Handshake.getsSession sid in
  let ch = sess.ch in
  let sh = sess.sh in
  (sh.server_version,sh.cipher_suite,sess.sid,ch.client_version,ch.cipher_suites)

let bldContentFromSessionInfo sid = 
  let version, cs, sid, maxver, csl = getSessionInfo sid in
  Printf.sprintf "\
      \n<table>\
      \n<tr><td><b>Version:</b> %s</td></tr>\
      \n<tr><td><b>Session id:</b> %s</td></tr>\
      \n<tr><td><b>Chosen ciphersuite:</b> %s</td></tr>\
      \n<tr><td><b>Client version offered:</b> %s</td></tr>\
      \n<tr><td><b>Client ciphersuites:</b> <br>&nbsp;&nbsp;%s</td></tr>\
      \n<tr><td><hr></td></tr>\
      \n</table>"
        (string_of_ProtocolVersion version) 
        (Data.sprint_raw (Bytearray.sub sid 1 (sid.Length - 1)))
        (string_of_bCipherSuite cs)
        (string_of_ProtocolVersion maxver) 
        (String.concat "<br>&nbsp;&nbsp;" (List.map string_of_CipherSuite csl)) 


  

let bldResponse body req =  
  let htmlheader = Printf.sprintf "<html>\n<body>\n<h1>Coucou! (%s)</h1>"  (System.DateTime.Now.ToString()) in
  let str = Printf.sprintf "Your HTTP header was:<br><br>\n\n\n%s</body></html>\n\n" (htmlize req) in
  let content = htmlheader^body^str in 
  let header = Printf.sprintf "\
      \nHTTP/1.1 200 OK\
      \nDate: %s\
      \nServer: BCFZ\
      \nContent-Length: %d\
      \nContent-Type: text/html; charset=ISO-8859-1\r\n\r\n"
        (System.DateTime.Now.ToString())
        content.Length 
  in header ^ content


let serverHttpResp net_conn connid sessionid = 
  let req = Record.recv connid in
  Printf.printf "Received the following request:\n-----\n%s\n-----\n" (Data.iutf8 req);
  let content = bldContentFromSessionInfo sessionid in
  let resp = bldResponse content (Data.iutf8 req) in
  Printf.printf "Sending the following response:\n-----\n%s\n-----\n" resp;
  Record.send connid (Data.utf8 resp);  
  Handshake.close connid false;
  Net.close net_conn

